(ns simplecron.core
    "A simple library to schedule task on one timer"
    (:use clojure.tools.trace)
    (:import java.util.Timer
             java.util.TimerTask))

(def timer (Timer. "Maintainance" true))

(defn- task
   "Creates a TimerTask object that executes the given function

   `f` A function to execute at a fixed rate"
   [name f]
   (proxy [java.util.TimerTask] []
          (getName ([] name))
          (run ([]
                (.println *err* (str "executing task: " name))
                (f)))))

(defn schedule
   "Schedules a new task on a timer"
   [taskdesc]
   (if-let [interval (:interval taskdesc)]
     (let [start (or (:start taskdesc) 0)
           name (:name taskdesc)
           taskfn (:fn taskdesc)]
       (.scheduleAtFixedRate timer
                           (task name taskfn)
                           start
                           interval))))

(defn seconds [s] (* s 1000))
(defn minutes [m] (seconds (* m 60)))
(defn hours [h] (minutes (* h 60)))
(defn days [d] (hours (* d 24)))

(defmacro deftask
  "Defines a new task and schedules it.

  `name` is the name of the task
  `config` is a hash map with config variables. Supported keys are
    :start (optional)   Milliseconds before the first start of the task.
                        If not provided, defaults to 0.
    :interval           Period in milliseconds after which the task should
                        be scheduled
  `body`                The body of the function"
  [name config & body]
  `(schedule
     (merge ~config
            {:name (name '~name)
            :fn (fn [] ~@body)})))
