(ns net.littleredcomputer.math.examples.central-potential
  (:refer-clojure :exclude [+ - * /])
  (:require [net.littleredcomputer.math.env :refer :all]
            [net.littleredcomputer.math.mechanics.lagrange :refer :all]))

(defn V-central
  "Potential energy of an object of mass M with state given by a local
   tuple in a 1/r^2 field generated by an object of mass m1 situated
   at (x1, y1)."
  [M m1 x1 y1]
  (let [c (up x1 y1)
        -Mm (- (* M m1))]
    (fn [[t x v]]
      (let [r (sqrt (square (- x c)))]
        (/ -Mm r)))))

(defn T-central
  "Kinetic energy of an object of mass M in free-fall with given local
  state (NB: the configuration of fixed masses do not contribute to the
  kinetic energy)"
  [M m1 x1 y1]
  (L-free-particle M))

(def L-central (- T-central V-central))

(defn periodic-drive
  [A ω φ]
  #(-> % (* ω) (+ φ) cos (* A)))

(defn central-state-derivative
  [M m1 x1 y1]
  (Lagrangian->state-derivative
    (L-central M m1 x1 y1)))

(defn evolver
  [t dt M x0 y0 xDot0 yDot0]
  (let [state-history (atom [])]
    ((evolve central-state-derivative
             M ;; object mass
             1000 ;; first planetary mass
             0 0  ;; first planetary x,y
             )
      (up 0.0
          (up x0 y0)
          (up xDot0 yDot0))
      (fn [t [_ q _]] (swap! state-history conj (into [t] q)))
      dt
      t
      1.0e-6
      {:compile true})
    @state-history))

(defn -main
  [& args]
  (println (simplify ((central-state-derivative 'M 'm 0 0) (up 0 (up 'x 'y) (up 'xdot 'ydot)))))
  (with-literal-functions [x y]
    (println (simplify (((Lagrange-equations (L-central 'M 'm 0 0)) (up x y)) 't))))
  (println (evolver 50 1/2 1 50 50 -2 0)))
