(ns net.molequedeideias.inga.rest
  (:require [edn-query-language.core :as eql]
            [clojure.string :as string]
            [dk.ative.docjure.spreadsheet :as excel]
            [ring.util.mime-type :as mime]
            [spec-coerce.core :as sc]
            [cheshire.core :as cheshire]
            [clojure.java.io :as io]
            [camel-snake-kebab.core :as csk]
            [com.wsscode.pathom.connect :as pc])
  (:import (java.net URLEncoder URLDecoder URI)
           (java.time Instant)
           (org.apache.cxf.attachment Rfc5987Util)))

(defn index-post
  [{:keys                          [path-params form-params headers]
    :net.molequedeideias.inga/keys [parser prefix]
    :as                            req}]
  (let [mutation (-> path-params :path symbol)
        tx `[{(~mutation ~(sc/coerce-structure form-params))
              [:net.molequedeideias.inga/next-page]}]
        result (parser req tx)
        goto (or (-> result (get mutation) :net.molequedeideias.inga/next-page)
                 (get headers "referer"))]
    (if goto
      {:headers {"Location"      (str
                                   (when (and prefix (not (.getHost (new URI goto))))
                                     (str "/" prefix))
                                   goto)
                 "Cache-Control" "no-store"}
       :status  303}
      {:headers {"Cache-Control" "no-store"}
       :status  202})))

(defn childrenfy
  [x]
  (cond
    (string? x) {:type         :prop
                 :dispatch-key (keyword x)
                 :key          (keyword x)}
    :else x))

(defn req->ast
  [{:keys [path-params query-params]}]
  (when (contains? path-params :path)
    (let [{:keys [ident-value children]} query-params
          children (if (string? children) [children] children)
          ident-key (-> path-params :path URLDecoder/decode keyword)
          has-childs? (not (empty? children))
          childs (when has-childs?
                   (mapv childrenfy children))
          ast (cond-> {:type         :prop
                       :dispatch-key ident-key
                       :key          ident-key}
                      (contains? query-params :ident-value) (assoc :key [ident-key ident-value])
                      has-childs? (assoc :type :join
                                         :query (eql/ast->query {:type     :root
                                                                 :children childs})
                                         :children childs))]

      ast)))

(defn slurp-pagination
  [{:net.molequedeideias.inga/keys [parser]
    :as                            req}
   {:keys [dispatch-key]
    :as   ast}
   data]
  (let [children [(update ast
                          :params merge (cond-> (select-keys data [:edn-query-language.pagination/elements-per-page
                                                                   :edn-query-language.pagination/current-page
                                                                   :edn-query-language.pagination/cursor])
                                                (contains? data :edn-query-language.pagination/current-page)
                                                (update :edn-query-language.pagination/current-page inc)))]
        {:edn-query-language.pagination/keys [edges]
         :as                                 raw-data} (get (some->> {:type     :root,
                                                                      :children children}
                                                                     (eql/ast->query)
                                                                     (parser req))
                                                            dispatch-key)]
    (if (empty? edges)
      (:edn-query-language.pagination/edges data)
      (slurp-pagination req ast (update raw-data :edn-query-language.pagination/edges (partial concat (:edn-query-language.pagination/edges data)))))))

(defn +pag
  [ast]
  (if (contains? ast :children)
    (update ast :children (fn [children]
                            (:children (eql/query->ast [{:edn-query-language.pagination/edges (eql/ast->query {:type     :root
                                                                                                               :children (mapv childrenfy children)})}
                                                        :edn-query-language.pagination/first-element-index
                                                        :edn-query-language.pagination/last-element-index
                                                        :edn-query-language.pagination/elements-per-page
                                                        :edn-query-language.pagination/current-page
                                                        :edn-query-language.pagination/cursor
                                                        :edn-query-language.pagination/total-count]))))
    ast))

(defn get-rest
  [{:net.molequedeideias.inga/keys [parser]
    :keys                          [query-params]
    :as                            req}]
  (let [accept (get query-params :content-type
                    (-> req :headers (get "accept" "")))
        {::pc/keys [indexes]} (parser req [::pc/indexes])
        params (into {}
                     (for [[k v] query-params
                           :when (qualified-ident? k)]
                       [k v]))
        {:keys [key dispatch-key]
         :as   ast} (cond-> (req->ast req)
                            (not (empty? params)) (assoc :params params))
        all-resolvers (-> indexes ::pc/index-oir (get dispatch-key) vals (->> (mapcat identity)))
        pag? (boolean (some #{:edn-query-language.pagination/elements-per-page
                              :edn-query-language.pagination/current-page
                              :edn-query-language.pagination/cursor}
                            (for [resolver all-resolvers
                                  param (get-in indexes [::pc/index-resolvers resolver ::pc/params])]
                              param)))
        raw-data (get (some->> (cond-> {:type     :root,
                                        :children [(if pag?
                                                     (+pag ast)
                                                     ast)]})

                               (eql/ast->query)
                               (parser req))
                      key)
        paginated? (when (map? raw-data)
                     (contains? raw-data :edn-query-language.pagination/edges))
        edn? (string/starts-with? accept "application/edn")
        xls? (string/starts-with? accept "application/vnd.ms-excel")
        data (cond
               (and xls? paginated?) (slurp-pagination req (if pag?
                                                             (+pag ast)
                                                             ast) raw-data)
               paginated? (:edn-query-language.pagination/edges raw-data)
               :else raw-data)
        filename (Rfc5987Util/encode (str (name (if (keyword? dispatch-key)
                                                  dispatch-key
                                                  (first dispatch-key)))
                                          "-"
                                          (subs (str (Instant/now))
                                                0 19)
                                          ".xls"))]
    (when data
      {:body    (cond
                  edn? (pr-str data)
                  xls? (let [data (if (map? data)
                                    [data]
                                    data)
                             ks (keys (first data))
                             getter (fn [x]
                                      (map str ((apply juxt ks) x)))
                             wb (excel/create-workbook (URLEncoder/encode (pr-str dispatch-key))
                                                       (concat [(map pr-str ks)]
                                                               (map getter data)))]
                         (fn [writer]
                           (try
                             (excel/save-workbook! (io/output-stream writer)
                                                   wb)
                             (catch Throwable e
                               (println e)))))
                  :else (fn [writer]
                          (cheshire/generate-stream data (io/writer writer))))
       :headers (into (cond-> {"Cache-Control" "no-store"
                               "Content-Type"  (cond
                                                 edn? (mime/default-mime-types "edn")
                                                 xls? (mime/default-mime-types "xls")
                                                 :else (mime/default-mime-types "json"))}
                              xls? (assoc "Content-Disposition"
                                          (str "attachment; filename*=\"" filename "\"; filename=\"" filename "\"")))
                      (when (map? raw-data)
                        (for [[k v] (dissoc raw-data :edn-query-language.pagination/edges)
                              :when (= "edn-query-language.pagination" (namespace k))]
                          [(csk/->HTTP-Header-Case-String (name k))
                           (str v)])))
       :status  200})))
