(ns net.ozias.swarm.swarm
  (:gen-class)
  (:require [clojure.java.io :as io]
            [clojure.pprint :refer (pprint)]
            [clojure.string :as str]
            [clojure.tools.cli :refer (parse-opts)]
            [clojure.tools.nrepl.server :refer (start-server stop-server)]
            [environ.core :refer :all]
            (net.ozias.swarm [config :refer (read-config)]
                             [libswarm :refer (version)])
            [org.ozias.cljlibs.bootstrap.bootstrap :as boot]
            [org.ozias.cljlibs.logging.logging :refer (configure-logging)]
            [taoensso.timbre :refer (info)]))

(def ^:private configuration (atom []))
(def ^:private nrepl-server (atom nil))
(def ^:private command-line-options
  [["-h" "--help" "Show the usage message"]
   [nil "--version" "Show version information"]
   ["-v" nil "Output verbosity level; may be specified multiple times to increase value"
    :id :verbosity
    :default 0
    :assoc-fn (fn [m k _] (update-in m [k] inc))]
   ["-r" "--repl" "Specify this option when running from a REPL"]
   ["-p" "--port" "Port for the nREPL server to listen on"
    :default 11579
    :parse-fn #(Integer/parseInt %)
    :validate [#(< 0 % 0x10000) "Must be a number between 0 and 65536"]]
   ["-f" "--configfile CONFIGFILE" "Configuration file path"
    :default (str (env :home) "/.swarmconfig")]
   [nil  "--logfile LOGFILE" "Log file path"
    :default (str (env :home) "/log/swarm.log")]])

(defn- usage [summary]
  (->> [(version)
        ""
        "Usage: swarm [options]"
        ""
        "Options:"
        summary
        ""]
       (str/join "\n")))

(defn- error-msg [errors]
  (str "The following errors occurred while parsing your command:\n\n"
       (str/join "\n" errors)))

(defn- merge-file-configuration [{:keys [options] :as opts}]
  (update-in opts [:options] merge (read-config (:configfile options))))

(defn- configure [command-line-arguments]
  (swap! configuration 
         (fn [_ command-line-arguments] 
           (-> command-line-arguments
               (parse-opts command-line-options)
               (merge-file-configuration)
               (configure-logging))) 
         command-line-arguments))

(defn- reset-config []
  (reset! configuration []))

(defn- start-nrepl-server [server & {:keys [port]}]
  (if (nil? server)
    (start-server :port port)
    server))

(defn- stop-nrepl-server [server]
  (when (not (nil? server))
    (stop-server server)
    nil))

(defn- stop-nrepl []
  (swap! nrepl-server stop-nrepl-server))

(defn- start-nrepl []
  (let [{{:keys [port]} :options} @configuration]
    (swap! nrepl-server start-nrepl-server :port port)))

(defn- exit [status msg {:keys [repl]}]
  (println msg)
  (if-not repl
    (System/exit status))
  status)

(defn bootstrap
  "Bootstrap a swarm instance at the given ip.

  Note that you must have ssh and scp successfully setup at that ip for
  this to work"
  [&{:keys [user host path jar options]}]
  (boot/bootstrap :user user :host host :path path :jar jar :options options)
  host)

(defn start 
  "Startup swarm.

  The options are parsed, the logging is configured, and the nREPL server is started.

  Use (startup \"--help\") at the REPL or 'swarm --help' and the command-line to see 
  valid options"
  [& args]
  (let [{:keys [options summary errors]} (configure args)]
    (cond
     (:help options) (exit 0 (usage summary) options)
     (:version options) (exit 0 (version) options)
     errors (exit 1 (error-msg errors) options)
     :else (do
             (start-nrepl)
             (info (version) "started succesfully")))))

(defn stop
  "Stop swarm.

  This includes shutting down the nREPL server and any background agents in not running from a
  repl.

  Use with caution!"
  []
  (let [{{:keys [repl]} :options} @configuration]
    (do
      (reset-config)
      (stop-nrepl)
      (when-not repl
        (shutdown-agents))
      (info (version) "shutdown successfully"))))

(defn restart
  "Restart the swarm.

  Useful for reloading .swarmconfig changes while running.

  You can also re-specify command-line options if needed."
  [& args]
  (reset-config)
  (let [config (configure args)]
    (and (seq config) (map? config))))

(defn status
  "swarm status"
  []
  (info (str "Configuration\n" (with-out-str (pprint @configuration))))
  (if (seq @nrepl-server)
    true
    false))

(defn -main
  "Command-line entry point."
  [& args]
  (apply start args))
