# clj-cukes

A pure Clojure implementation of Cucumber, for behaviour driven development.

[![Clojars Project](https://img.shields.io/clojars/v/new-ways/clj-cukes.svg)](https://clojars.org/new-ways/clj-cukes)

## Rationale

While a Clojure compatible implementation of Cucumber-JVM exists, it is a)
difficult to use, b) poorly documented, and c) does not support a functional
programming style, in the sense that all state must be stored in mutable global
variables making concurrent testing a lot harder than it needs to be.

Therefore we wanted a Cucumber test runner which supports threading of state
through each step, and is easy to use from within Clojure code itself.

This project is still in early Alpha, use with caution! Feel free to open an
issue for any bugs you find or features you'd like added, or contribute your
own changes with a pull or merge request.

## Usage

```clojure
(require '[new-ways.clj-cukes :refer :all])
(require '[clojure.test :refer :all])

;; Define steps with the defstep function, which takes a regex for matching 
;; your gherkin steps to and a function to execute.
;; The function will be passed the current state (a map) as its first argument
;; followed by the matching groups from the regex

(defstep #"^I make a (\S+) request for (\S+)$"
  (fn make-request [state method path]
    (let [method-kw (-> method .toLowerCase keyword)]
      (assoc state ::response ((app) (request method-kw path))))))

;; There are also 5 macros for convenience: Given, When, Then, And, and But
;; These call defstep under the hood, but use a shorter syntax:

(When #"^I make a (\S+) request for (\S+)$" [state method path]
  (let [method-kw (-> method .toLowerCase keyword)]
    (assoc state ::response ((app) (request method-kw path))))))

(Then #"^The status code should be (\d+)$" [state status-code]
  (is (= (Integer/parseInt status-code) (:status (::response state))))))


;; Add an individual gherkin file to your test suite
(deftest handler
  (run-gherkin-file "test/features/handler.feature"))

;; Or run all the feature files in a folder (recursively)
(run-gherkin-files "test/features")

;; Or simply run some gherkin represented as data
(execute-gherkin
  [:feature "App responds to HTTP requests with the right response code"
   [:scenario "Public routes should receive 200"
    [:when "I make a GET request for /login"]
    [:then "The status code should be 200"]]])

```

## Development

1. Start a REPL with `clj`
2. Run tests once with `clj -A:test:runner`
3. Start a REPL with test dependencies with `clj -A:test`
4. Create a pom.xml (just if you need it for the Cursive IDE) with `clj -S:pom`


## License

Copyright © 2018 New Ways

Distributed under the Eclipse Public License either version 1.0 or (at
your option) any later version.
