// Compiled by ClojureScript 1.9.562 {}
goog.provide('nicheware.platform.utilities.common.version');
goog.require('cljs.core');
goog.require('nicheware.platform.utilities.common.core');
/**
 * Makes a function that will test an element with a reference key item against the name (where name
 * will match the first item of the ref-key value, i.e. the asset-key)
 * 
 * - ref-key: The key in the element that contains the reference.
 * - name: The value of the name part (or asset-key) of the reference that the returned function will check
 * - returns: ```fn(element)``` where element should have a key of ref-key that is a reference  ```[asset-key version]``` and the function will return true if the reference asset-key equals "name"
 */
nicheware.platform.utilities.common.version.make_ref_name_fn = (function nicheware$platform$utilities$common$version$make_ref_name_fn(ref_key,name){
return (function (elem){
return cljs.core._EQ_.call(null,cljs.core.first.call(null,cljs.core.get.call(null,elem,ref_key)),name);
});
});
/**
 * Search's a vector for the first element that has a version reference matching the given name.
 * 
 *   Uses ref-key to determine which item in the element map is the version reference, then matches
 *   name against the first item of this reference.
 * 
 *   Useful when searching for references to a versioned asset.
 * 
 *   - vec: Vector of elements. ```[ {<ref-key> [<asset-name> <asset-version] ...}, ...]```
 *   - ref-key: The key within each element in the vector that contains the version reference of the form ```[<asset-key> <version>]```
 *   - name: The asset key to be matched.
 *   - returns: The full element from vec which has a reference with that name.
 */
nicheware.platform.utilities.common.version.find_element_with_ref_name = (function nicheware$platform$utilities$common$version$find_element_with_ref_name(vec,ref_key,name){
return nicheware.platform.utilities.common.core.find_by_pred.call(null,vec,nicheware.platform.utilities.common.version.make_ref_name_fn.call(null,ref_key,name));
});
/**
 * Searches a vector of elements where each element holds a reference ```[name, version]```. It will
 * search for the element that references the given name, and replace it with the supplied replacement element.
 * 
 * - vec: Vector of elements to search. ```[ {<ref-key> [name, version] ...}, ...]```
 * - replacement-element: New element to replace the matching element.
 * - ref-key: The key in the element that holds the reference ```[name/asset-key, version]```
 * - name: The name/asset-key in the reference that is being searched for.
 * - returns: New vector with the suitable replacement element.
 *   
 */
nicheware.platform.utilities.common.version.replace_element_with_ref_name = (function nicheware$platform$utilities$common$version$replace_element_with_ref_name(vec,replacement_element,ref_key,name){
return nicheware.platform.utilities.common.core.replace_element_by_pred.call(null,vec,replacement_element,nicheware.platform.utilities.common.version.make_ref_name_fn.call(null,ref_key,name));
});
/**
 * Extracts an ordered list of versions (from least to highest modified time) from the versioned-asset.
 * 
 * - versioned-asset: Versioned asset to be searched for versions.  ```{:versions {<mod-key-1> {<asset-1>}}}```
 * - returns: vector of versions (from smallest to highest) ```[<asset-1> ...]```]
 */
nicheware.platform.utilities.common.version.get_ordered_versions = (function nicheware$platform$utilities$common$version$get_ordered_versions(p__9814){
var map__9817 = p__9814;
var map__9817__$1 = ((((!((map__9817 == null)))?((((map__9817.cljs$lang$protocol_mask$partition0$ & (64))) || ((cljs.core.PROTOCOL_SENTINEL === map__9817.cljs$core$ISeq$)))?true:false):false))?cljs.core.apply.call(null,cljs.core.hash_map,map__9817):map__9817);
var versioned_asset = map__9817__$1;
var versions = cljs.core.get.call(null,map__9817__$1,new cljs.core.Keyword(null,"versions","versions",536521978));
return cljs.core.sort_by.call(null,new cljs.core.Keyword(null,"modified-time","modified-time",891335115),cljs.core.vals.call(null,versions));
});
/**
 * Treats the versions within the versioned-assets as a vector of version modification times, from smallest to highest,
 * and returns the index within that vector of the provided elements modified time.
 * 
 * - versioned-asset: Versioned asset to be searched for versions.  ```{:versions {<mod-key-1> {<asset-1>}}}```
 * - element: Uses the ```:modified-time``` attribute as the value to find the index of.
 * - returns index or nil if not found.
 * 
 * 
 */
nicheware.platform.utilities.common.version.find_index = (function nicheware$platform$utilities$common$version$find_index(versioned_asset,p__9820){
var map__9823 = p__9820;
var map__9823__$1 = ((((!((map__9823 == null)))?((((map__9823.cljs$lang$protocol_mask$partition0$ & (64))) || ((cljs.core.PROTOCOL_SENTINEL === map__9823.cljs$core$ISeq$)))?true:false):false))?cljs.core.apply.call(null,cljs.core.hash_map,map__9823):map__9823);
var element = map__9823__$1;
var modified_time = cljs.core.get.call(null,map__9823__$1,new cljs.core.Keyword(null,"modified-time","modified-time",891335115));
return nicheware.platform.utilities.common.core.find_index_by_pred.call(null,nicheware.platform.utilities.common.version.get_ordered_versions.call(null,versioned_asset),((function (map__9823,map__9823__$1,element,modified_time){
return (function (p1__9819_SHARP_){
return cljs.core._EQ_.call(null,new cljs.core.Keyword(null,"modified-time","modified-time",891335115).cljs$core$IFn$_invoke$arity$1(p1__9819_SHARP_),modified_time);
});})(map__9823,map__9823__$1,element,modified_time))
);
});
/**
 * Gets the current version key for the specified asset
 * 
 *   - assets: Map of assets, keyed by assets key, where each asset is versioned. ```{<asset-1-key> {:current <mod-time> :versions {:mod-time-1 <asset-v1> ...}}}```
 *   - asset-key: Key into the asset map
 *   - returns: The ```:current``` of the asset matching the given key. nil if asset not present.
 */
nicheware.platform.utilities.common.version.current_version_key = (function nicheware$platform$utilities$common$version$current_version_key(assets,asset_key){
return cljs.core.get_in.call(null,assets,new cljs.core.PersistentVector(null, 2, 5, cljs.core.PersistentVector.EMPTY_NODE, [asset_key,new cljs.core.Keyword(null,"current","current",-1088038603)], null));
});
/**
 * Build an asset reference [name version] for the named asset using the current version
 * 
 *   - assets: Maps of assets, keyed by assets key, where each asset is versioned. ```{<asset-1-key> {:current <mod-time> :versions {:mod-time-1 <asset-v1> ...}}}```
 *   - asset-key: Key into the asset map
 *   - returns: A reference to the current version of the asset. Format is ```[asset-key version]```. version is usually modified time.
 *  if asset not in assets, reference will have nil as the version.
 */
nicheware.platform.utilities.common.version.get_current_ref = (function nicheware$platform$utilities$common$version$get_current_ref(assets,asset_key){
return new cljs.core.PersistentVector(null, 2, 5, cljs.core.PersistentVector.EMPTY_NODE, [asset_key,nicheware.platform.utilities.common.version.current_version_key.call(null,assets,asset_key)], null);
});
/**
 * Access the first asset in the asset map and then extracts the reference to the current version
 * for that asset
 * 
 *   - assets: Maps of assets, keyed by assets key, where each asset is versioned. ```{<asset-1-key> {:current <mod-time> :versions {:mod-time-1 <asset-v1> ...}}}```
 *   - returns: A reference to the current version of the first asset. Format is ```[asset-key version]```. version is usually modified time.
 */
nicheware.platform.utilities.common.version.get_first_asset_ref = (function nicheware$platform$utilities$common$version$get_first_asset_ref(assets){
var vec__9828 = cljs.core.first.call(null,assets);
var asset_key = cljs.core.nth.call(null,vec__9828,(0),null);
var _ = cljs.core.nth.call(null,vec__9828,(1),null);
return nicheware.platform.utilities.common.version.get_current_ref.call(null,assets,asset_key);
});
/**
 * Gets the specified version of the named asset
 * 
 *   - assets: Maps of assets, keyed by assets key, where each asset is versioned. ```{<asset-1-key> {:current <mod-time> :versions {:mod-time-1 <asset-v1> ...}}}```
 *   - reference: Vector of the asset-key (or name) and the required version (usually modified-time). ```[asset-key version]```
 *   - returns: the asset with the required key and version.
 */
nicheware.platform.utilities.common.version.get_version = (function nicheware$platform$utilities$common$version$get_version(assets,p__9831){
var vec__9835 = p__9831;
var asset_key = cljs.core.nth.call(null,vec__9835,(0),null);
var version = cljs.core.nth.call(null,vec__9835,(1),null);
return cljs.core.get_in.call(null,assets,new cljs.core.PersistentVector(null, 3, 5, cljs.core.PersistentVector.EMPTY_NODE, [asset_key,new cljs.core.Keyword(null,"versions","versions",536521978),version], null));
});
/**
 * Gets the current version of the asset with the given key.
 * 
 *   - assets: Maps of assets, keyed by asset key, where each asset is versioned. ```{<asset-1-key> {:current <mod-time> :versions {:mod-time-1 <asset-v1> ...}}}```
 *   - asset-key: Key of the asset for which the current version required.
 *   - returns: The current version of the asset.
 */
nicheware.platform.utilities.common.version.get_current = (function nicheware$platform$utilities$common$version$get_current(assets,asset_key){
return nicheware.platform.utilities.common.version.get_version.call(null,assets,new cljs.core.PersistentVector(null, 2, 5, cljs.core.PersistentVector.EMPTY_NODE, [asset_key,nicheware.platform.utilities.common.version.current_version_key.call(null,assets,asset_key)], null));
});
/**
 * Gets the current version of the given versioned asset.
 * 
 * - versioned-asset: Versioned asset to be searched for versions.  ```{:current <mod-time> :versions {<mod-key-1> {<asset-1>}}}```
 * - returns: The current version of the asset.
 */
nicheware.platform.utilities.common.version.get_current_asset = (function nicheware$platform$utilities$common$version$get_current_asset(p__9838){
var map__9841 = p__9838;
var map__9841__$1 = ((((!((map__9841 == null)))?((((map__9841.cljs$lang$protocol_mask$partition0$ & (64))) || ((cljs.core.PROTOCOL_SENTINEL === map__9841.cljs$core$ISeq$)))?true:false):false))?cljs.core.apply.call(null,cljs.core.hash_map,map__9841):map__9841);
var versioned_asset = map__9841__$1;
var current = cljs.core.get.call(null,map__9841__$1,new cljs.core.Keyword(null,"current","current",-1088038603));
var versions = cljs.core.get.call(null,map__9841__$1,new cljs.core.Keyword(null,"versions","versions",536521978));
return cljs.core.get.call(null,versions,current);
});
/**
 * Renames element in the versions map, modifying the ```:name``` attribute to the new name.
 * 
 * - versions: Versions map, each entry to have name updated.  ```{<mod-key-1> {:name <asset-key> ...}, <mod-key-2> {..} ...}```
 * - new-name: New name to use for the asset, and hence for each version.
 * - returns: updated version map where the name in the versions are modified to the new name.
 */
nicheware.platform.utilities.common.version.rename_versions = (function nicheware$platform$utilities$common$version$rename_versions(versions,new_name){
return nicheware.platform.utilities.common.core.map_values.call(null,(function (p1__9843_SHARP_){
return cljs.core.assoc.call(null,p1__9843_SHARP_,new cljs.core.Keyword(null,"name","name",1843675177),new_name);
}),versions);
});
/**
 * Will change the top level name of the asset and change the name in all the versions of the asset.
 * 
 * - versioned-asset: Versioned asset to be searched for versions.  ```{:current <mod-time> :name <asset-key> :versions {<mod-key-1> {:name <asset-key> ...}, <mod-key-2> {..} ...}}```
 * - new-name: New name to use for the asset, and hence for each version.
 * - returns: updated versioned-asset map where the top level name and the name in the versions are modified to the new name.
 */
nicheware.platform.utilities.common.version.rename_versioned_asset = (function nicheware$platform$utilities$common$version$rename_versioned_asset(p__9844,new_name){
var map__9847 = p__9844;
var map__9847__$1 = ((((!((map__9847 == null)))?((((map__9847.cljs$lang$protocol_mask$partition0$ & (64))) || ((cljs.core.PROTOCOL_SENTINEL === map__9847.cljs$core$ISeq$)))?true:false):false))?cljs.core.apply.call(null,cljs.core.hash_map,map__9847):map__9847);
var versioned_asset = map__9847__$1;
var versions = cljs.core.get.call(null,map__9847__$1,new cljs.core.Keyword(null,"versions","versions",536521978));
return cljs.core.assoc.call(null,cljs.core.assoc.call(null,versioned_asset,new cljs.core.Keyword(null,"name","name",1843675177),new_name),new cljs.core.Keyword(null,"versions","versions",536521978),nicheware.platform.utilities.common.version.rename_versions.call(null,versions,new_name));
});
/**
 * Marks the element with the appropriate modification time.
 * 
 *   - element: Map in which a new ```:modified-time``` keyword to be added, if required.
 *   - force: optional 2 arity argument. If true will update ```:modified-time``` even if it exists. false by default.
 *   - returns: map with ```:modified-time``` set to current time in millis if not set or force = true.
 */
nicheware.platform.utilities.common.version.timestamp_element = (function nicheware$platform$utilities$common$version$timestamp_element(var_args){
var args9849 = [];
var len__8628__auto___9852 = arguments.length;
var i__8629__auto___9853 = (0);
while(true){
if((i__8629__auto___9853 < len__8628__auto___9852)){
args9849.push((arguments[i__8629__auto___9853]));

var G__9854 = (i__8629__auto___9853 + (1));
i__8629__auto___9853 = G__9854;
continue;
} else {
}
break;
}

var G__9851 = args9849.length;
switch (G__9851) {
case 1:
return nicheware.platform.utilities.common.version.timestamp_element.cljs$core$IFn$_invoke$arity$1((arguments[(0)]));

break;
case 2:
return nicheware.platform.utilities.common.version.timestamp_element.cljs$core$IFn$_invoke$arity$2((arguments[(0)]),(arguments[(1)]));

break;
default:
throw (new Error([cljs.core.str.cljs$core$IFn$_invoke$arity$1("Invalid arity: "),cljs.core.str.cljs$core$IFn$_invoke$arity$1(args9849.length)].join('')));

}
});

nicheware.platform.utilities.common.version.timestamp_element.cljs$core$IFn$_invoke$arity$1 = (function (element){
return nicheware.platform.utilities.common.version.timestamp_element.call(null,element,false);
});

nicheware.platform.utilities.common.version.timestamp_element.cljs$core$IFn$_invoke$arity$2 = (function (element,force){
if(cljs.core.truth_((function (){var or__7407__auto__ = force;
if(cljs.core.truth_(or__7407__auto__)){
return or__7407__auto__;
} else {
return new cljs.core.Keyword(null,"modified-time","modified-time",891335115).cljs$core$IFn$_invoke$arity$1(element);
}
})())){
return element;
} else {
return cljs.core.assoc.call(null,element,new cljs.core.Keyword(null,"modified-time","modified-time",891335115),nicheware.platform.utilities.common.core.current_time_millis.call(null));
}
});

nicheware.platform.utilities.common.version.timestamp_element.cljs$lang$maxFixedArity = 2;

/**
 * Prepare a new element for versioning. Will ensure timestamped and also
 * not starred (as is new version)
 * 
 * - element: Map which represent the version of an asset to be placed into list of versions.
 * - returns: same element, but with appropriate ```:modified-time``` and no ```:starred``` attribute.
 */
nicheware.platform.utilities.common.version.version_element = (function nicheware$platform$utilities$common$version$version_element(element){
var pred_val__9440__auto__ = ((cljs.core.fn_QMARK_.call(null,new cljs.core.Keyword(null,"starred","starred",-601770202)))?new cljs.core.Keyword(null,"starred","starred",-601770202).cljs$core$IFn$_invoke$arity$1(nicheware.platform.utilities.common.version.timestamp_element.call(null,element)):new cljs.core.Keyword(null,"starred","starred",-601770202));
if(cljs.core.truth_(pred_val__9440__auto__)){
return ((function (pred_val__9440__auto__){
return (function (p1__9856_SHARP_){
return cljs.core.assoc.call(null,p1__9856_SHARP_,new cljs.core.Keyword(null,"starred","starred",-601770202),false);
});})(pred_val__9440__auto__))
.call(null,nicheware.platform.utilities.common.version.timestamp_element.call(null,element));
} else {
return nicheware.platform.utilities.common.version.timestamp_element.call(null,element);
}
});
/**
 * Adds to the version list, if not already in the list, will append to the end.
 * Becomes the new current version.
 * 
 * - versioned-asset: Existing versioned asset, with map of current versions. ```{:current <mod-key-n> :versions {<mod-key-1> {<asset-1> ...}}}```
 * - element: New element to add to the list. Will set the ```:modified-time``` to be current time if not already set
 * - returns: versioned-asset with element added as the latest version (i.e. ```:current``` updated)
 */
nicheware.platform.utilities.common.version.add_version = (function nicheware$platform$utilities$common$version$add_version(versioned_asset,element){
var map__9859 = nicheware.platform.utilities.common.version.version_element.call(null,element);
var map__9859__$1 = ((((!((map__9859 == null)))?((((map__9859.cljs$lang$protocol_mask$partition0$ & (64))) || ((cljs.core.PROTOCOL_SENTINEL === map__9859.cljs$core$ISeq$)))?true:false):false))?cljs.core.apply.call(null,cljs.core.hash_map,map__9859):map__9859);
var versioned_element = map__9859__$1;
var modified_time = cljs.core.get.call(null,map__9859__$1,new cljs.core.Keyword(null,"modified-time","modified-time",891335115));
return cljs.core.assoc.call(null,(cljs.core.truth_(cljs.core.get_in.call(null,versioned_asset,new cljs.core.PersistentVector(null, 2, 5, cljs.core.PersistentVector.EMPTY_NODE, [new cljs.core.Keyword(null,"versions","versions",536521978),modified_time], null)))?versioned_asset:cljs.core.assoc_in.call(null,versioned_asset,new cljs.core.PersistentVector(null, 2, 5, cljs.core.PersistentVector.EMPTY_NODE, [new cljs.core.Keyword(null,"versions","versions",536521978),modified_time], null),versioned_element)),new cljs.core.Keyword(null,"current","current",-1088038603),modified_time);
});
/**
 * Replaces the current version with the supplied version, but does not change any version number.
 * 
 *   - versioned-asset: Existing versioned asset, with map of current versions. ```{:current <mod-key> :versions {<mod-key-1> {<asset-1>}}}```
 *   - asset: New asset to replace the existing current.
 *   - returns: Updated versioned-asset.
 * 
 *   Should be used only when updates are to be made to the current version which you don't want to affect versioning.
 */
nicheware.platform.utilities.common.version.replace_current = (function nicheware$platform$utilities$common$version$replace_current(p__9861,asset){
var map__9864 = p__9861;
var map__9864__$1 = ((((!((map__9864 == null)))?((((map__9864.cljs$lang$protocol_mask$partition0$ & (64))) || ((cljs.core.PROTOCOL_SENTINEL === map__9864.cljs$core$ISeq$)))?true:false):false))?cljs.core.apply.call(null,cljs.core.hash_map,map__9864):map__9864);
var versioned_asset = map__9864__$1;
var current = cljs.core.get.call(null,map__9864__$1,new cljs.core.Keyword(null,"current","current",-1088038603));
var updated_asset = cljs.core.assoc.call(null,asset,new cljs.core.Keyword(null,"modified-time","modified-time",891335115),current);
return cljs.core.assoc_in.call(null,versioned_asset,new cljs.core.PersistentVector(null, 2, 5, cljs.core.PersistentVector.EMPTY_NODE, [new cljs.core.Keyword(null,"versions","versions",536521978),current], null),updated_asset);
});
/**
 * Sets the current version to the ```:modified-time``` of the given asset. Assumes this exists already as a version.
 * 
 *   - versioned-asset: Existing versioned asset, with map of current versions. ```{:current <mod-key> :versions {<mod-key-1> {<asset-1>}}}```
 *   - asset: New asset to whose modified-time is to become the current version.
 *   - returns: Updated versioned-asset.
 */
nicheware.platform.utilities.common.version.set_version = (function nicheware$platform$utilities$common$version$set_version(versioned_asset,asset){
return cljs.core.assoc.call(null,versioned_asset,new cljs.core.Keyword(null,"current","current",-1088038603),new cljs.core.Keyword(null,"modified-time","modified-time",891335115).cljs$core$IFn$_invoke$arity$1(asset));
});
/**
 * Updates the version by either replacing the current or adding a new version.
 * The action taken depends on whether the current one has the same mutation as passed in. If the same
 * we dont bother creating a new version and just replace current.
 * 
 * - mutation: The data to be added to the asset using the ```:mutation``` key.
 * - versioned-asset: Existing versioned asset, with map of current versions. ```{:current <mod-key> :versions {<mod-key-1> {<asset-1>}}}```
 * - asset: asset data to have mutation added.
 * - returns: updated versioned-asset with the mutated asset added as a new and latest version, or if the current assest has the same mutation,
 *   just replaced the current version with the new asset data.
 */
nicheware.platform.utilities.common.version.mutate_version = (function nicheware$platform$utilities$common$version$mutate_version(mutation,versioned_asset,asset){
var updated_asset = cljs.core.assoc.call(null,asset,new cljs.core.Keyword(null,"mutation","mutation",-285823378),mutation);
var current_mutation = new cljs.core.Keyword(null,"mutation","mutation",-285823378).cljs$core$IFn$_invoke$arity$1(nicheware.platform.utilities.common.version.get_current_asset.call(null,versioned_asset));
if(cljs.core._EQ_.call(null,mutation,current_mutation)){
return nicheware.platform.utilities.common.version.replace_current.call(null,versioned_asset,updated_asset);
} else {
return nicheware.platform.utilities.common.version.add_version.call(null,versioned_asset,updated_asset);
}
});
/**
 * Removes the specified version from the versioned asset. Ensures at least one asset version remains, (i.e, you cannot delete the last version) and that if the
 * deleted asset is current a new asset (first) is selected as current.
 * 
 *   - versioned-asset: Existing versioned asset, with map of current versions. ```{:current <mod-key> :versions {<mod-key-1> {<asset-1>}}}```
 *   - version: The version to be removed, typically the modified time.
 *   - returns: updated versioned-asset.
 */
nicheware.platform.utilities.common.version.delete_version = (function nicheware$platform$utilities$common$version$delete_version(p__9866,version){
var map__9870 = p__9866;
var map__9870__$1 = ((((!((map__9870 == null)))?((((map__9870.cljs$lang$protocol_mask$partition0$ & (64))) || ((cljs.core.PROTOCOL_SENTINEL === map__9870.cljs$core$ISeq$)))?true:false):false))?cljs.core.apply.call(null,cljs.core.hash_map,map__9870):map__9870);
var versioned_asset = map__9870__$1;
var current = cljs.core.get.call(null,map__9870__$1,new cljs.core.Keyword(null,"current","current",-1088038603));
var versions = cljs.core.get.call(null,map__9870__$1,new cljs.core.Keyword(null,"versions","versions",536521978));
var updated_versions = cljs.core.dissoc.call(null,versions,version);
if((cljs.core.count.call(null,versions) > (1))){
var G__9872 = versioned_asset;
var G__9872__$1 = cljs.core.assoc.call(null,G__9872,new cljs.core.Keyword(null,"versions","versions",536521978),updated_versions)
;
if(cljs.core._EQ_.call(null,current,version)){
return cljs.core.assoc.call(null,G__9872__$1,new cljs.core.Keyword(null,"current","current",-1088038603),cljs.core.first.call(null,cljs.core.first.call(null,updated_versions)));
} else {
return G__9872__$1;
}
} else {
return versioned_asset;
}
});
/**
 * Same as delete version execpt it is given the version to be deleted rather than just
 * the version number (modified-time).
 * 
 * If the version is the current one, then current becomes on the remaining versions.
 * 
 *   - versioned-asset: Existing versioned asset, with map of current versions. ```{:current <mod-key> :versions {<mod-key-1> {<asset-1-data>}}}```
 *   - asset: Asset to be removed. Must include a ```:modified-time``` key, to identify the version to be deleted.
 *   - returns: updated versioned-asset, with version removed and ```:current``` updated if the current was deleted.
 */
nicheware.platform.utilities.common.version.delete_asset_version = (function nicheware$platform$utilities$common$version$delete_asset_version(versioned_asset,p__9873){
var map__9876 = p__9873;
var map__9876__$1 = ((((!((map__9876 == null)))?((((map__9876.cljs$lang$protocol_mask$partition0$ & (64))) || ((cljs.core.PROTOCOL_SENTINEL === map__9876.cljs$core$ISeq$)))?true:false):false))?cljs.core.apply.call(null,cljs.core.hash_map,map__9876):map__9876);
var asset = map__9876__$1;
var modified_time = cljs.core.get.call(null,map__9876__$1,new cljs.core.Keyword(null,"modified-time","modified-time",891335115));
return nicheware.platform.utilities.common.version.delete_version.call(null,versioned_asset,modified_time);
});
/**
 * Removes any unused versions. A used version is either current or marked as starred (has a ```:starred``` true attribute) or returns true to the optional given predicate function.
 * 
 *   - versioned-asset: Existing versioned asset, with map of current versions. ```{:current <mod-1> :versions {<mod-key-1 {:starred <bool> <asset-1-data>}}}```
 *   - pred-fn:  optional last argument should accept the version and return true if used. Defaults to a function that is always false
 *   - returns: versioned-asset with unused versions removed.
 */
nicheware.platform.utilities.common.version.remove_unused_versions = (function nicheware$platform$utilities$common$version$remove_unused_versions(var_args){
var args9879 = [];
var len__8628__auto___9885 = arguments.length;
var i__8629__auto___9886 = (0);
while(true){
if((i__8629__auto___9886 < len__8628__auto___9885)){
args9879.push((arguments[i__8629__auto___9886]));

var G__9887 = (i__8629__auto___9886 + (1));
i__8629__auto___9886 = G__9887;
continue;
} else {
}
break;
}

var G__9881 = args9879.length;
switch (G__9881) {
case 1:
return nicheware.platform.utilities.common.version.remove_unused_versions.cljs$core$IFn$_invoke$arity$1((arguments[(0)]));

break;
case 2:
return nicheware.platform.utilities.common.version.remove_unused_versions.cljs$core$IFn$_invoke$arity$2((arguments[(0)]),(arguments[(1)]));

break;
default:
throw (new Error([cljs.core.str.cljs$core$IFn$_invoke$arity$1("Invalid arity: "),cljs.core.str.cljs$core$IFn$_invoke$arity$1(args9879.length)].join('')));

}
});

nicheware.platform.utilities.common.version.remove_unused_versions.cljs$core$IFn$_invoke$arity$1 = (function (versioned_asset){
return nicheware.platform.utilities.common.version.remove_unused_versions.call(null,versioned_asset,cljs.core.constantly.call(null,false));
});

nicheware.platform.utilities.common.version.remove_unused_versions.cljs$core$IFn$_invoke$arity$2 = (function (p__9882,pred_fn){
var map__9883 = p__9882;
var map__9883__$1 = ((((!((map__9883 == null)))?((((map__9883.cljs$lang$protocol_mask$partition0$ & (64))) || ((cljs.core.PROTOCOL_SENTINEL === map__9883.cljs$core$ISeq$)))?true:false):false))?cljs.core.apply.call(null,cljs.core.hash_map,map__9883):map__9883);
var versioned_asset = map__9883__$1;
var current = cljs.core.get.call(null,map__9883__$1,new cljs.core.Keyword(null,"current","current",-1088038603));
var versions = cljs.core.get.call(null,map__9883__$1,new cljs.core.Keyword(null,"versions","versions",536521978));
var remove_pred = ((function (map__9883,map__9883__$1,versioned_asset,current,versions){
return (function (p1__9878_SHARP_){
return cljs.core.not.call(null,(function (){var or__7407__auto__ = cljs.core._EQ_.call(null,current,new cljs.core.Keyword(null,"modified-time","modified-time",891335115).cljs$core$IFn$_invoke$arity$1(p1__9878_SHARP_));
if(or__7407__auto__){
return or__7407__auto__;
} else {
var or__7407__auto____$1 = new cljs.core.Keyword(null,"starred","starred",-601770202).cljs$core$IFn$_invoke$arity$1(p1__9878_SHARP_);
if(cljs.core.truth_(or__7407__auto____$1)){
return or__7407__auto____$1;
} else {
return pred_fn.call(null,p1__9878_SHARP_);
}
}
})());
});})(map__9883,map__9883__$1,versioned_asset,current,versions))
;
return cljs.core.assoc.call(null,versioned_asset,new cljs.core.Keyword(null,"versions","versions",536521978),nicheware.platform.utilities.common.core.filter_remove_val.call(null,remove_pred,versions));
});

nicheware.platform.utilities.common.version.remove_unused_versions.cljs$lang$maxFixedArity = 2;


//# sourceMappingURL=version.js.map