(ns nicolasoury.updatable-map-reduce.map-reduce
  (:import java.util.concurrent.ConcurrentHashMap
	   [clojure.lang IPersistentCollection
	    ITransientCollection Counted IEditableCollection ILookup
	    IMeta IObj Seqable IPersistentList IPersistentMap IPersistentStack
	    IPersistentSet IPersistentVector
	    Associative
	    ITransientAssociative ITransientMap ITransientVector
	    ITransientSet
	    ] 
	   )
  (:gen-class)) 

(def default-history-depth  5)

(set! *warn-on-reflection* true)

(defprotocol Reduce
  (zero [x] "The neutral element of the reduction")
  (plus [x y z] "The actual reduction")
  )


(defprotocol InvertibleReduce
   (minus [x y z] "minus x x = zero. plus x (minus z x) = z.")
   (invertible-helper [x] "Returns a MapReduceHelper if it needs one")
   )

  
(defprotocol
  MapReduce
  (map-reduce [x data] "Apply the map reduce x on the data")
)

(declare invertible-map-reducable)


(defprotocol MapReduceHelper
  (go-on? [helper content positive_diff negative_diff] "Checks whether we should go on to update or reduce from scratch.
Returns nil if we need to reduce form scratch. returns a MapReduceHelper for next round (maybe itself) if we need to go on update") 
  (why-not? [helper content positive_diff negative_diff remapped-nil] "Explains why we should stop. Returns nil to recompute everything and anything else if we know the result
                                                          from something else. In which case the result is direcly cached. Use remapped-nil for nil")
 )

(deftype InfiniteGoOnHelper []
  MapReduceHelper
  (go-on? [x _ _ _] x)
  )

(def infinite-go-on-helper (InfiniteGoOnHelper.))
  
(defmacro map-reducer
  "MapReducer creates an map-reducer objects. You must gives it afew options imcluding:
 - :map default identity, expressed as a term of arg
 - :plus, the reulsting term of plus, in tem of arg1 and arg2
 - :zero the value of zero, default nil
 - : minus optional, expressed in term of arg1 arg2 "
  [& args]
  (let [options-map (apply hash-map args)]
	`(reify 
	 Reduce
	 (zero [_#] ~(:zero options-map))
	 (plus [_# ~'arg1 ~'arg2] ~(:plus options-map))
	 ~@(if (:minus options-map)
	     `(InvertibleReduce
	       (minus [_# ~'arg1 ~'arg2] ~(:minus options-map))
	       (invertible-helper [x] ~(or (:invertible-helper options-map) `infinite-go-on-helper ))
	       )
	     ())
	 MapReduce
	 (map-reduce [~'this ~'data]
		     ~(or
		       (:reduction options-map)
		       `(loop [l# (seq ~'data) res# (zero ~'this)]
			 (if (empty? l#)
			   res#
			   (let [~'arg (first l#)
				 elt# ~(or (:map options-map) 'arg)]
			     (recur (rest l#) (plus ~'this res# elt#))))))
	 ))))

(defmacro map-reduce-fn [& options]
  `(let [mr-object# (map-reducer ~@options)]
     (fn [s#] (map-me-and-reduce s# mr-object#))))

(defprotocol MapReducable
  (map-me-and-reduce [data mr] "use map and reducer mr on data. Can use some particularities of data and mr or call (map-reduce mr data)"))




(extend-type Object
  MapReducable
  (map-me-and-reduce [x mr] (map-reduce mr x)))

(def remapped-nil :nicolasoury.invertible-map-reduce.map-reduce/nil)

(defmacro map-nil [expr]
  `(let [res# ~expr]
     (if (nil? res#) remapped-nil res#)))


(defmacro unmap-nil [expr]
  `(let [res# ~expr]
     (if (identical? res# remapped-nil) nil res#)))
  
(defprotocol
  Caching
  (cached [x f] "Returns the cache value of f on x, or nil.")
  (cache [x f v] "Cache v as the value of f. Map nil if necessary"))

(defprotocol
  Diff
  (forget-ancestor [x])
  (ancestor [x] "Returns the ancestor from which the diff is taken")
  (positive-diff [x] "Returns the diff element")
  (negative-diff [x] "Returns the diff element")
  )

(declare transient-invertible-map-reducable)

(declare invertible-map-reducable)

(defn  prune-history [^InvertibleMapReducable x  history-depth  my-depth]
  (let [next-depth (+ (int my-depth) (int 1)) history-depth (int history-depth)]
    (if (>= next-depth history-depth)
      (do
	(loop [i (int my-depth) ^InvertibleMapReducable data x]
	  (if (<= i 0)
	    (forget-ancestor data)
	    (if-let [anc (ancestor data)]
	      (recur (- i (int 1)) anc))))
	 0)
       next-depth)))

(deftype InvertibleMapReducable
  [^int history-depth
   ^int my-depth
   ^ConcurrentHashMap my_cache
   my_content
   my_positive_diff
   my_negative_diff
   ^{:unsynchronized-mutable true} ^InvertibleMapReducable my_ancestor]
  Caching
  (cached [x f]  (if-let [res (.get my_cache f)] (unmap-nil res)))
  (cache [x f v]   (do (.put my_cache f (map-nil v)) v ))
  Diff
  (forget-ancestor [x] (set! my_ancestor nil))
  (ancestor [x]  my_ancestor)
  (positive_diff [x] my_positive_diff)
  (negative_diff [x] my_negative_diff)
  MapReducable
  (map-me-and-reduce [data mr]
		     (loop [current-data data current-positive-diff (zero mr) current-negative-diff (zero mr) current-helper (invertible-helper mr)]
		       (if-let [next-helper (go-on? current-helper current-data current-positive-diff current-negative-diff)]
			 (if-let [v (cached current-data mr)]  ;; TOO SLOW : we recache at each read...
			   (cache data mr (minus mr (plus mr v  current-positive-diff) current-negative-diff))
			   (if-let [new-ancestor (ancestor current-data)]
			     (recur new-ancestor
				    (plus mr current-positive-diff (map-me-and-reduce (positive-diff current-data) mr))
				    (plus mr current-negative-diff (map-me-and-reduce (negative-diff current-data) mr))
				    next-helper)
			     (cache data mr (map-me-and-reduce my_content mr))))
			 ;; why de we stop
			 (if-let [result (why-not? current-helper current-data current-positive-diff current-negative-diff remapped-nil)]
			     (cache data mr (unmap-nil result))
			     (cache data mr (map-me-and-reduce my_content mr))))))


  IPersistentCollection
  (empty [_] (invertible-map-reducable (.empty ^IPersistentCollection my_content)))
  (equiv [_ o] (.equiv ^IPersistentCollection my_content o))
  (cons [this x]
	(cond
	 (set? my_content)
	   (if (contains? my_content x) this 
	       (InvertibleMapReducable. history-depth (prune-history this history-depth my-depth)
					(ConcurrentHashMap.)
					(conj my_content x)
					[x] []  this))
	 (and (map? my_content) (map? x))
	    (loop [things (seq x) positive_diff {} negative_diff {}]
	      (if things
		(let [[a b] (first things)]
		  (if-let [old (get my_content a)]
		    (recur (next things) (assoc positive_diff a b) (assoc negative_diff a old))
		    (recur (next things) (assoc positive_diff a b)  negative_diff)))
		(InvertibleMapReducable. history-depth (prune-history this history-depth my-depth)
		                           (ConcurrentHashMap.) (conj my_content x) positive_diff negative_diff this)))
	 (map? my_content)
	       (assoc this (first x) (second x))
	 (vector? my_content)
	     (.assocN this (count my_content) x)
	 true
	 (InvertibleMapReducable. history-depth (prune-history this history-depth my-depth)
				  (ConcurrentHashMap.)
				  (conj my_content x)
				  [x] []  this))  
	)
  Counted
  (count [x] (count my_content))
  IEditableCollection
  (asTransient [x] (transient-invertible-map-reducable history-depth my-depth (transient my_content)

						       (transient []) (transient []) x))

  ILookup
  (valAt [_ key] (.valAt ^ILookup my_content key))
  (valAt [_ key default] (.valAt ^ILookup my_content key default))
  IMeta
  (meta [_] (.meta ^IMeta my_content))
  IObj
  (withMeta [_ x] (InvertibleMapReducable. history-depth my-depth my_cache (.withMeta ^IObj my_content x) my_positive_diff my_negative_diff my_ancestor))
  Seqable
  (seq [_] (seq my_content))
  IPersistentList
  IPersistentStack
  (peek [_](.peek ^IPersistentStack my_content ))
  (pop [this] (let [val (.peek ^IPersistentStack my_content )]
		(InvertibleMapReducable.  history-depth (prune-history this history-depth my-depth)
		                      (ConcurrentHashMap.)
				      (.pop  ^IPersistentStack my_content)
				      nil [val] this)))
  
  IPersistentMap
  (without [this key]
	   (if (contains? my_content key)
	     (InvertibleMapReducable.  history-depth (prune-history this history-depth my-depth)
				       (ConcurrentHashMap.) (.without ^IPersistentMap my_content key)
				       []  [[key (get my_content key)]] this)
	      this
	      ))
  (assocEx [this key val]
	   (InvertibleMapReducable.  history-depth (prune-history this history-depth my-depth)
	     (ConcurrentHashMap.) (.assocEx ^IPersistentMap my_content key val)
			       [[key val]]  (if-let [old-val (get my_content key)] [[key old-val]] nil)  this))
  Associative
  (containsKey [_ key] (.containsKey ^Associative my_content key))
  (entryAt [_ key] (.entryAt ^Associative my_content key))
  (assoc [this key val]
    (cond
     (map? my_content)
     (InvertibleMapReducable.  history-depth (prune-history this history-depth my-depth)
       (ConcurrentHashMap.) (assoc my_content key val)
				 [[key val]]  (if-let [old-val (get my_content key)] [[key old-val]] nil)  this)
     (vector? my_content)
     (InvertibleMapReducable.  history-depth (prune-history this history-depth my-depth)
                                     (ConcurrentHashMap.) (assoc my_content key val)           
				 [val]  (if-let [old-val (get my_content key)] [old-val] nil) this)))
  IPersistentSet
  (disjoin [this key] (if (contains? my_content key)
			(InvertibleMapReducable.  history-depth (prune-history this history-depth my-depth) (ConcurrentHashMap.) (disj my_content key)
						 []  [key] this)
			this))
  (contains [this ke] (contains? my_content key))
  (get [this key] (.get ^IPersistentSet my_content key))
  IPersistentVector
  (assocN [this i v]  (let [new-content (.assocN ^IPersistentVector my_content i v)]
			(if (> (count new-content) (count my_content))
			      (InvertibleMapReducable.  history-depth (prune-history this history-depth my-depth) (ConcurrentHashMap.) new-content
						 [v]  []  this)
			      (InvertibleMapReducable. history-depth (prune-history this history-depth my-depth)  (ConcurrentHashMap.) new-content
						 [v]  [(get my_content i)] this)
			      )))
   )


(extend-type nil
  MapReducable
  (map-me-and-reduce [x mr] (zero mr)))



(defn invertible-map-reducable
  ([content  history-depth]
     (InvertibleMapReducable. (int history-depth) 0 (ConcurrentHashMap.) content nil nil nil))
  ([content]
     (InvertibleMapReducable. (int default-history-depth) 0 (ConcurrentHashMap.) content nil nil nil)))
  



(deftype TransientInvertibleMapReducable
  [^int history-depth ^int ancestor-depth
   ^{:unsynchronized-mutable true} ^ITransientCollection content
   ^{:unsynchronized-mutable true} ^ITransientVector positive_diff
   ^{:unsynchronized-mutable true} ^ITransientVector negative_diff
   ancestor]
  ITransientCollection
  (conj [this x]
	(cond
	 (instance? ITransientSet content)
	   (if (contains? content x) this  ;; WARNING Clojure is BUGGED
	       (do
		 (set! content (conj! content x))
		 (set! positive_diff (conj! positive_diff x))
		 this))
	 (instance? ITransientMap content)
	     (assoc! this (first x) (second x))
	 (instance? ITransientVector content)
	     (.assocN this (count content) x)
	 true
	      (do
		 (set! content (conj! content x))
		 (set! positive_diff (conj! positive_diff x))
		 this)))

  (persistent [this]
	      (InvertibleMapReducable. history-depth (prune-history ancestor history-depth ancestor-depth)
				     (ConcurrentHashMap.)
				    (persistent! content) (persistent! positive_diff)
				    (persistent! negative_diff) ancestor))
  Counted
  (count [_] (count content))

  ITransientMap
  (without [this key] 
  	   (if-let  [v (get content key)]
	     (do
	       (set! content (dissoc! content key))
	       (set! negative_diff (conj! negative_diff [key v]))
	       )) 
	      this
	      )
  ITransientAssociative
  (assoc [this key val]
    (let [old  (get content key)]
      (cond
       (instance? ITransientMap content)
        (do
	  (set! content (assoc! content key val))
	  (set! positive_diff (conj! positive_diff [key val]))
	  (when old  (set! negative_diff (conj! negative_diff [key old]))))
       (instance? ITransientVector content)
         (do
	  (set! content (assoc! content key val))
	  (set! positive_diff (conj! positive_diff val))
	  (when old  (set! negative_diff (conj! negative_diff old))))
	 ))
    this)
  ITransientVector
  (assocN [this key val] (.assoc ^ITransientAssociative this key val))
  (pop [this]
       (set! negative_diff (conj! negative_diff (get content (- (count content) 1))))
       (set! content (pop! content)))
  ITransientSet
  (disjoin [this key] (when true ;(contains? this key) contains is BUGGY
			(set! content (disj! content key))
			(set! negative_diff (conj! negative_diff key)))
	   this)
  
  (contains [this key] (.contains ^ITransientSet content key))
  (get [this key] (.get ^ITransientSet content key))

  
  )
(defn transient-invertible-map-reducable [i j c x y z]
  (TransientInvertibleMapReducable. i j c x y z))


(defmethod print-method InvertibleMapReducable [x y] (print-method (seq x) y))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;; EXAMPLES
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(def mr-plus (map-reducer :plus (+ arg1 arg2) :minus (- arg1 arg2) :zero 0))
(def a (invertible-map-reducable (range 1 1000000)))
(def b (invertible-map-reducable (range 1 100)))
(def mr-times (map-reducer :plus (* arg1 arg2) :minus (/ arg1 arg2) :zero 1
			   :invertible-helper (reify MapReduceHelper
						     (go-on? [x _ pos neg] (if (zero? (* pos neg)) nil x))
						     (why-not? [_ pos neg _ _] (if (zero? neg) nil 0)))

			   ))


(defn test-plus []
  (time (map-me-and-reduce a mr-plus))
  (time (map-me-and-reduce (conj a 15) mr-plus)))


(defn test-times []
  (time (map-me-and-reduce (conj a 0)  mr-times))
  (time (map-me-and-reduce a mr-times))
 nil
  )


(def updatable-map-reducable-class InvertibleMapReducable)

(def updatable-empty-map (invertible-map-reducable {}))
