(ns nl.jomco.openapi.v3.coercer
  (:require
   [clojure.string :as string]
   [nl.jomco.openapi.v3.json-pointer :as pointer])
  (:import
   (java.util.regex Pattern)))

(defn- split-simple
  [s]
  (string/split s #","))

(defn- simple-exploder
  [s]
  (if (string? s)
    [s]
    s))

(defn default-style
  [in]
  (case in
    "path"   "simple"
    "query"  "form"
    "header" "simple"
    "cookie" "form"))

(defn default-expode
  [in]
  (case in
    "path"   false
    "query"  true
    "header" true
    "cookie" true))

(defn- mk-splitter
  "Return a splitter that converts the input string to a vector of
  strings."
  [in style explode? n]
  (let [style (or style (default-style in))
        explode? (if (some? explode?)
                   explode?
                   (default-expode in))]
    (case style
      "simple"
      split-simple

      "label"
      (let [sep (if explode?  #"," #"\.")]
        (fn [s]
          (when (string/starts-with? s ".")
            (-> s
                (subs 1)
                (string/split s sep)))))

      "matrix"
      (let [sep (if explode?
                  (Pattern/quote (str ";" (name n) "="))
                  #",")]
        (fn [s]
          (when (string/starts-with? s (str ";" (name n) "="))
            (-> s
                (subs (+ 2 (count (name n))))
                (string/split sep)))))

      "form"
      (if explode?
        simple-exploder
        ;; multiple items
        #(string/split % #","))

      "spaceDelimited"
      (if explode?
        simple-exploder
        ;; multiple items
        #(string/split % #" "))

      "pipeDelimited"
      (if explode?
        simple-exploder
        ;; multiple items
        #(string/split % #"\|")))))

(defn- mk-object-parser
  "Return an object parser that converts the input string to a map
  of strings."
  [in style explode? n]
  (let [style    (or style (default-style in))
        explode? (if (some? explode?)
                   explode?
                   (default-expode in))]
    (case style
      "simple"
      (if explode?
        (fn [s]
          (into {} (partition 2 (string/split s #","))))
        (fn [s]
          (into {} (->> (string/split s #",")
                        (map #(string/split % #"="))))))
      "label"
      (if explode?
        (fn [s]
          (into {} (partition 2 (-> s
                                    (string/replace #"^\." "")
                                    (string/split #",")))))
        (fn [s]
          (into {} (->> (string/split s #"\.")
                        next ;; starts with a dot
                        (map #(string/split % #"="))))))

      "matrix"
      (if explode?
        (let [prefix (Pattern/quote (str ";" (name n) "="))]
          (fn [s]
            (into {} (->> (-> s
                              (string/replace prefix "")
                              (string/split #";"))
                          (map #(string/split % #"="))))))
        (fn [s]
          (-> s
              (->> (string/split s #";")
                   next
                   (map #(string/split % #"="))))))

      "form"
      (if explode?
        ;; FIXME -- need access to whole params object
        (throw (ex-info "Can't explode form object" {}))
        (fn [s]
          (into {} (partition 2 (string/split s ",")))))

      ;; TODO: deepObject
      )))

(defn identity-coercer
  [instance]
  [instance nil])

(defn- simple-parser
  "Return a parser for a simple (non-collection) `type`"
  [type]
  (case type
    "integer"
    parse-long

    "number"
    parse-double

    "boolean"
    parse-boolean

    (nil "string")
    identity

    nil))

(defn- mk-simple-coercer
  "Return a coercer for a simple parsable `type`

  otherwise returns nil."
  [type canonical-schema-path]
  (when-let [parser (simple-parser type)]
    (fn simple-coercer [instance path schema-path]
      (when instance ;; TODO: ??
        (if-let [instance (parser instance)]
          [instance nil]
          [nil [{:canonical-schema-path canonical-schema-path
                 :instance              instance
                 :issue                 "coercion-error"
                 :path                  path
                 :schema                {"type" type}
                 :schema-path           schema-path}]])))))

;; Coercers take an instance and instance path (like a validator) but
;; return a result pair (vector): a parsed instance and a collection
;; of issues.

(defn schema-coercer
  [{:keys [specification]} schema-path]
  (when-let [[schema-path schema] (pointer/find specification
                                                     schema-path)]
    (mk-simple-coercer (get schema "type" "string") schema-path)))

(defn parameter-coercer
  [{:keys [specification] :as context} canonical-schema-path]
  (when-let [[canonical-schema-path {:strs [style explode name in]}]
             (pointer/find specification canonical-schema-path false)]
    (assert (string? in))
    (let [schema    (-> specification
                        (pointer/get (conj canonical-schema-path "schema")))
          root-type (get schema "type" "string")]
      (or (mk-simple-coercer root-type canonical-schema-path)
          (case root-type
            "array"
            (let [split  (mk-splitter in style explode name)
                  coerce (schema-coercer context
                                         (into canonical-schema-path
                                               ["schema" "items"]))]
              (fn array-coercer [instance path schema-path]
                (let [items     (split instance)
                      coercions (->> items
                                     (map-indexed
                                      #(coerce %2 (conj path %1) (conj schema-path ["schema" "items"]))))]
                  [(mapv first coercions)
                   (mapcat second coercions)])))

            "object" ;; TODO - deepObject, spaceDelimited,
            ;; pipeDelimited and form explode=true
            (let [parser (mk-object-parser in style explode name)
                  coerce (schema-coercer context
                                         (conj canonical-schema-path ["schema"]))]
              (fn object-coercer [instance path schema-path]
                (coerce (parser instance) path schema-path))))))))

;; TODO: refactor common code with parameter-coercer
(defn header-coercer ;; for responses
  [{:keys [specification] :as context} canonical-schema-path name]
  (when-let [[canonical-schema-path {:strs [style explode]}]
             (pointer/find specification canonical-schema-path false)]
    (let [schema    (-> specification
                        (pointer/get (conj canonical-schema-path :schema)))
          root-type (get schema "type" "string")]
      (or (mk-simple-coercer root-type canonical-schema-path)
          (case root-type
            "array"
            (let [split  (mk-splitter "header" style explode name)
                  coerce (schema-coercer context
                                         (into canonical-schema-path
                                               ["schema" "items"]))]
              (fn array-coercer [instance path schema-path]
                (let [items     (split instance)
                      coercions (->> items
                                     (map-indexed
                                      #(coerce %2 (conj path %1) (conj schema-path ["schema-items"]))))]
                  [(mapv first coercions)
                   (mapcat second coercions)])))

            "object" ;; TODO - deepObject, spaceDelimited,
            ;; pipeDelimited and form explode=true
            (let [parser (mk-object-parser "header" style explode name)
                  coerce (schema-coercer context
                                         (conj canonical-schema-path ["schema"]))]
              (fn object-coercer [instance path schema-path]
                (coerce (parser instance) path schema-path))))))))
