(ns nl.jomco.openapi.v3.json-pointer
  "Parse and resolve JSON Pointers"
  (:require [clojure.string :as string])
  (:refer-clojure :exclude [deref get-in get find]))

(defn as-key
  "Coerce `x` to key (string or integer)."
  [x]
  (if (and (string? x)
            (re-matches #"-?\d+" x))
    (parse-long x)
    x))

(defn- key-path
  "Coerce path components to keys. See `as-key`."
  [path]
  (mapv as-key path))

(defn pointer-path
  "Parse a JSON Pointer as a path vector.

  The returned path's keys are integers or strings."
  [pointer]
  (cond
    (string/starts-with? pointer "#/")
    (-> pointer
        (subs 2)
        (string/split #"/")
        key-path)
    :else
    (throw (ex-info (str "Can't parse json pointer " pointer)
                    {:pointer pointer}))))

(defn as-path
  "Coerce `x` to a path. If `x` is a string, parse as JSON pointer."
  [x]
  (if (sequential? x)
    x
    (pointer-path x)))

(defn find
  "Find node and canonical path in `document`.

  Given a document that may contain $ref pointers, and a path, return
  a tuple (vector) of the canonical path and the node at that
  path. Throws an exception when cyclic references are encountered.

  If `path` is a string, it will be parsed as a JSON pointer. The
  returned path is always a vector.

  Returns nil when path does not point to a node in the document. If
  `throw-on-nil?` is provided and true, raise an error instead.

  See also `canonical-path` and `pointer-path`."
  ([document path throw-on-nil?]
   {:pre [(some? document) path]}
   (loop [node           document
          node-path      path ;; path from the current node
          canonical-path path ;; path from the root
          resolved       []
          seen           #{}]
     (cond
       (clojure.core/get node "$ref")
       (let [pp (pointer-path (clojure.core/get node "$ref"))]
         (if (seen pp)
           (throw (ex-info (str "Cyclic $refs found dereferencing "
                                (pr-str path))
                           {:resolved  resolved
                            :seen      seen
                            :last-seen pp
                            :path      path}))
           (let [path (into pp node-path)]
             (recur document
                    path
                    path
                    resolved
                    (conj seen pp)))))

       (nil? node)
       (when throw-on-nil?
         (throw (ex-info (str "Path " (pr-str canonical-path) " resolves to nil")
                         {:path           path
                          :resolved       resolved
                          :canonical-path canonical-path}))
         nil)

       (empty? node-path)
       [canonical-path node]

       :else
       (let [k (first node-path)]
         (recur (clojure.core/get node k)
                (next node-path)
                canonical-path
                (conj resolved k)
                seen)))))
  ([document path]
   (find document path false)))

(defn canonical-path
  "Get the canonical version of `path` in `document`.

  Given a document that may contain $ref pointers, and a path, return
  the equivalent path that contains no pointers. Throws an exception
  when cyclic references are encountered.

  Returns nil when path does not point to a node in the document. If
  `throw-on-nil?` is provided and true, raise an error instead.

  See also `find` and `get`."
  ([document path throw-on-nil?]
   (first (find document path throw-on-nil?)))
  ([document path]
   (canonical-path document path false)))

(defn get
  "Get node at `path` in `document.

  Given a document that may contain $ref pointers, and a path, return
  the equivalent path that contains no pointers. Throws an exception
  when cyclic references are encountered.

  Returns nil when path does not point to a node in the document. If
  `throw-on-nil?` is provided and true, raise an error instead.

  See also `find` and `get`."

  ([document path throw-on-nil?]
   (second (find document path throw-on-nil?)))
  ([document path]
   (get document path false)))
