(ns nl.jomco.openapi.v3.validator.json-schema-validator
  (:require
   [nl.jomco.openapi.v3.json-pointer :as pointer]
   [nl.jomco.openapi.v3.util :refer [cached-at! delayedfn]]))

(defn unique?
  [items]
  (loop [seen  (transient #{})
         items items]
    (if (seq items)
      (if (contains? seen (first items))
        false
        (recur (conj! seen (first items)) (next items)))
      true)))

(defn validator-context
  [specification]
  {:specification specification
   :cache         (atom {})})

(defn combine-issues
  [i1 i2]
  (if (and i1 i2)
    (into i1 i2)
    (or i1 i2)))

(defn checks
  "Combine a collection of checks taking the same instance and path into
  a single check."
  [coll]
  (reduce (fn [res check]
            (if (and res check)
              (fn [instance instance-path schema-path]
                (combine-issues (res instance instance-path schema-path)
                                (check instance instance-path schema-path)))
              (or res check)))
          (constantly nil)
          coll))

(defmulti key-validator
  "Compile a check for the given JSON Schema key.

  Path must be a canonical path to an existing key"
  (fn [_context path]
    (last path))
  :default nil)

(defmethod key-validator nil
  [_context _canonical-schema-path]
  (fn [_instance _instance-path _schema-path] nil))

(defn with-schema-path-section
  [validator schema-path-section]
  {:pre [(sequential? schema-path-section)]}
  (fn [instance path schema-path]
    (validator instance path (into schema-path schema-path-section))))

(defn validation-error
  [{:keys [path schema-path canonical-schema-path schema] :as error}]
  {:pre [path schema-path canonical-schema-path
         (contains? error :instance)
         (some? schema)]}
  (assoc error
         :issue "schema-validation-error"))

(defn keyword-error
  [{:keys [specification]}
   {:keys [path schema schema-path canonical-schema-path] :as error}]
  {:pre [(some? specification) path schema-path canonical-schema-path]}
  (let [schema-keyword (last canonical-schema-path)]
    (-> error
        (assoc :schema-keyword schema-keyword
               :schema (assoc schema
                              schema-keyword (get-in specification canonical-schema-path)))
        validation-error)))

(defn schema-validator
  [{:keys [specification cache] :as context} canonical-schema-path]
  {:pre [context (some? specification) canonical-schema-path]}
  (let [[canonical-schema-path schema]
        (pointer/find specification canonical-schema-path true)]
    (cond
      ;; boolean schema's always fail or always succeed
      (true? schema)
      (constantly nil)

      (false? schema)
      (fn [instance path schema-path]
        [(validation-error {:instance              instance
                            :path                  path
                            :canonical-schema-path canonical-schema-path
                            :schema-path           schema-path
                            :schema                schema})])

      (map? schema)
      ;; we need a delay since schemas can be defined
      ;; recursively (properties of a schema can use the
      ;; parent schema). This enables us to compile the
      ;; schema validator only once, the first time the
      ;; validator is invoked
      (->> (delayedfn
            (->> (keys schema)
                 (map #(-> (key-validator context
                                          (conj canonical-schema-path %))
                           (with-schema-path-section [%])))
                 checks))
           ;; we cache the validator in the context so we can reuse
           ;; the same schema validator when reached from multiple
           ;; `$ref`s
           (cached-at! cache [::schema-validators canonical-schema-path]))

      :else
      (throw (ex-info (str "Can't parse schema" (pr-str schema))
                      {:schema                schema
                       :canonical-schema-path canonical-schema-path})))))

(defn- mk-check
  [{:keys [specification] :as context} canonical-schema-path check-fn]
  {:pre [context specification canonical-schema-path (fn? check-fn)]}
  (let [schema-val (pointer/get specification canonical-schema-path true)
        check*     (check-fn schema-val)]
    (fn check [instance instance-path schema-path]
      (when-let [error-props (check* instance instance-path schema-path)]
        (assert (map? error-props)
                (pr-str [schema-path check-fn error-props]))
        [(keyword-error context
                        (merge {:instance              instance
                                :path                  instance-path
                                :schema-path           schema-path
                                :canonical-schema-path canonical-schema-path}
                               error-props))]))))

(defn- mk-pred
  [context canonical-schema-path pred-fn]
  {:pre [context canonical-schema-path pred-fn]}
  (mk-check context canonical-schema-path
            (fn [schema-val]
              (let [p (pred-fn schema-val)]
                (when-not p
                  (throw (ex-info "`(pred-fn schema-val)` did not return a function"
                                  {:pred-fn               pred-fn
                                   :canonical-schema-path canonical-schema-path
                                   :schema-val            schema-val})))
                (fn pred-check [instance _ _]
                  (let [res (p instance)]
                    (assert (boolean? res)
                            (str "Non-bool predicate result for " canonical-schema-path))
                    (when-not res
                      {})))))))

(defn- type-pred
  "Type t is a string or a collection of strings"
  [t]
  (cond
    (string? t)
    (case t
      "string"  string?
      "array"   sequential?
      "boolean" boolean?
      "integer" integer?
      "null"    nil?
      "number"  number?
      "object"  map?)

    (sequential? t)
    (fn [instance]
      (boolean (some #(% instance) (map type-pred t))))))

(defmethod key-validator "type"
  [context canonical-schema-path]
  (mk-pred context canonical-schema-path type-pred))

(defmethod key-validator "maxLength"
  [context path]
  (mk-pred context path
           (fn [max-length]
             (fn [instance]
               (if (string? instance)
                 (<= (count instance) max-length)
                 true)))))

(defmethod key-validator "minLength"
  [context path]
  (mk-pred context path
           (fn [min-length]
             (fn [instance]
               (if (string? instance)
                 (>= (count instance) min-length)
                 true)))))

(defmethod key-validator "pattern"
  [context path]
  (mk-pred context path
           (fn [pattern]
             (let [regex (java.util.regex.Pattern/compile pattern)]
               (fn [instance]
                 (if (string? instance)
                   (boolean (re-find regex instance))
                   true))))))

(def ^:private uuid-re
  #"^[0-9a-fA-F]{8}\b-[0-9a-fA-F]{4}\b-[0-9a-fA-F]{4}\b-[0-9a-fA-F]{4}\b-[0-9a-fA-F]{12}$")

(defmethod key-validator "format"
  [context path]
  (mk-pred context path
           (fn [format]
             (case format
               ;; TODO: add more formats
               "uuid" (fn [s]
                        (if (string? s)
                          (boolean (re-find uuid-re s))
                          true))
               ;; default ok
               (fn [_]
                 true)))))

(defmethod key-validator "multipleOf"
  [context path]
  (mk-pred context path
           (fn [multiple-of]
             {:pre [(pos? multiple-of)]}
             (fn [instance]
               (if (number? instance)
                 (zero? (rem instance multiple-of))
                 true)))))

(defn sibling-path
  [path k]
  (assoc path (dec (count path)) k))

(defmethod key-validator "maximum"
  [{:keys [specification] :as context} path]
  (let [exclusive? (-> specification
                       (get-in (sibling-path path "exclusiveMaximum")))
        cmp        (if exclusive? < <=)
        err-props  (if (some? exclusive?)
                     {:schema {"exclusiveMaximum" exclusive?}}
                     {})]
    (mk-check context path
              (fn [maximum]
                {:pre [(number? maximum)]}
                (fn [instance _ _]
                  (when (and (number? instance)
                             (not (cmp instance maximum)))
                    err-props))))))

(defmethod key-validator "minimum"
  [{:keys [specification] :as context} path]
  (let [exclusive? (-> specification
                       (get-in (sibling-path path "exclusiveMinimum")))
        cmp        (if exclusive? > >=)
        err-props  (if (some? exclusive?)
                     {:schema {"exclusiveMinimum" exclusive?}}
                     {})]
    (mk-check context path
              (fn [minimum]
                {:pre [(number? minimum)]}
                (fn [instance _ _]
                  (when (and (number? instance)
                             (not (cmp instance minimum)))
                    err-props))))))

;; TODO: Check for impact on "unevaluatedItems"; section 10.3.1.3

(defmethod key-validator "contains"
  [{:keys [specification] :as context} canonical-schema-path]
  (let [min-contains (-> specification
                         (get-in (sibling-path canonical-schema-path "minContains")))
        max-contains (-> specification
                         (get-in (sibling-path canonical-schema-path "maxContains")))
        validator    (schema-validator context canonical-schema-path)
        sub-issues   (fn [instance path schema-path]
                       (assert (sequential? instance))
                       (->> instance
                            (map-indexed (fn [i el]
                                           (validator el (conj path i) schema-path)))
                            vec))]
    (if (= 0 min-contains) ;; may be nil, so can't use `zero?`
      (constantly nil)
      (mk-check context canonical-schema-path
                (fn [_]
                  (cond
                    (and min-contains max-contains)
                    (fn [instance path schema-path]
                      (when (sequential? instance)
                        (let [sub-issues (sub-issues instance path schema-path)
                              count      (count (filter nil? sub-issues))]
                          (when-not (<= min-contains
                                        count
                                        max-contains)
                            {:schema     {"minContains" min-contains
                                          "maxContains" max-contains}
                             :hints      {:contains-count count}
                             :sub-issues sub-issues}))))

                    max-contains
                    (fn [instance path schema-path]
                      (when (sequential? instance)
                        (let [sub-issues (sub-issues instance path schema-path)
                              count      (count (filter nil? sub-issues))]
                          (when-not (<= count max-contains)
                            {:schema {"maxContains" max-contains}
                             :hints  {:contains-count count}
                             :sub-issues sub-issues}))))

                    min-contains
                    (fn [instance path schema-path]
                      (when (sequential? instance)
                        (let [sub-issues (sub-issues instance path schema-path)
                              count      (count (filter nil? sub-issues))]
                          (when-not (<= min-contains count)
                            {:schema {"minContains" min-contains}
                             :hints  {:contains-count count}
                             :sub-issues sub-issues}))))

                    :else
                    (fn [instance path schema-path]
                      (when (sequential? instance)
                        (let [sub-issues (sub-issues instance path schema-path)
                              count      (count (filter nil? sub-issues))]
                          (when-not (pos? count)
                            {:hints {:contains-count count}
                             :sub-issues sub-issues}))))))))))

(defmethod key-validator "items"
  [{:keys [specification] :as context} canonical-schema-path]
  (let [canonical-schema-path (pointer/canonical-path specification canonical-schema-path true)
        validator             (schema-validator context canonical-schema-path)]
    (fn [instance path schema-path]
      (when (sequential? instance)
        (loop [res   nil
               i     0
               items instance]
          (if-let [[item & rst] (seq items)]
            (recur (combine-issues res (validator item (conj path i) schema-path))
                   (inc i)
                   rst)
            res))))))

(defmethod key-validator "maxItems"
  [context canonical-schema-path]
  (mk-pred context canonical-schema-path
           (fn [max-items]
             (fn [instance]
               (if (sequential? instance)
                 (<= (count instance) max-items)
                 true)))))

(defmethod key-validator "minItems"
  [context canonical-schema-path]
  (mk-pred context canonical-schema-path
           (fn [min-items]
             (fn [instance]
               (if (sequential? instance)
                 (>= (count instance) min-items)
                 true)))))

(defmethod key-validator "uniqueItems"
  [context canonical-schema-path]
  (mk-pred context canonical-schema-path
           (fn [unique]
             (if unique
               (fn [instance]
                 (if (sequential? instance)
                   (unique? instance)
                   true))
               (constantly true)))))

(defmethod key-validator "allOf"
  [{:keys [specification] :as context} canonical-schema-path]
  (let [[canonical-schema-path all-of] (pointer/find specification canonical-schema-path)]
    (->> (range (count all-of))
         (map #(-> (schema-validator context (conj canonical-schema-path %))
                   (with-schema-path-section [%])))
         (checks))))

(defmethod key-validator "oneOf"
  [{:keys [specification] :as context} canonical-schema-path]
  (let [[canonical-schema-path one-of] (pointer/find specification canonical-schema-path)
        validators                     (->> (range (count one-of))
                                            (mapv #(schema-validator context (conj canonical-schema-path %))))]
    (fn [instance path schema-path]
      (let [sub-issues (vec (map-indexed (fn [i validator]
                                            (validator instance path (conj schema-path i)))
                                          validators))
            ok-count   (count (filter nil? sub-issues))]
        (when-not (=  1 ok-count)
          [(keyword-error context
                          {:canonical-schema-path canonical-schema-path
                           :hints                 {:ok-count ok-count}
                           :instance              instance
                           :path                  path
                           :schema-path           schema-path
                           :sub-issues            sub-issues})])))))

(defmethod key-validator "anyOf"
  [{:keys [specification] :as context} canonical-schema-path]
  (let [[canonical-schema-path any-of] (pointer/find specification canonical-schema-path)
        validators                     (->> (range (count any-of))
                                            (mapv #(-> (schema-validator context (conj canonical-schema-path %))
                                                       (with-schema-path-section [%])))
                                            (apply juxt))]
    (fn [instance path schema-path]
      (let [sub-issues (validators instance path schema-path) ;; returns vector of vector of issues
            ok-count   (count (filter nil? sub-issues))]
        (when-not (<  0 ok-count)
          [(keyword-error context
                          {:canonical-schema-path canonical-schema-path
                           :hints                 {:ok-count ok-count}
                           :instance              instance
                           :path                  path
                           :schema-path           schema-path
                           :sub-issues            sub-issues})])))))

(defmethod key-validator "properties"
  [{:keys [specification] :as context} canonical-schema-path]
  {:pre [specification context canonical-schema-path]}
  (let [[canonical-schema-path properties] (pointer/find specification canonical-schema-path)
        validators (->> (keys properties)
                        (map #(vector % (schema-validator context (conj canonical-schema-path %))))
                        (into {}))]
    (fn properties-check
      [instance path schema-path]
      (when (map? instance)
        (reduce-kv (fn [ret prop-key validator]
                     (if-let [[_ v] (find instance prop-key)]
                       (combine-issues ret (validator v
                                                      (conj path prop-key)
                                                      (conj schema-path prop-key)))
                       ret))
                   nil
                   validators)))))

(defmethod key-validator "maxProperties"
  [context canonical-schema-path]
  (mk-check context canonical-schema-path
            (fn [max-properties]
              (fn [instance _ _]
                (when (map? instance)
                  (let [cnt (count (keys instance))]
                    (when (> cnt max-properties)
                      {:count cnt})))))))


(defmethod key-validator "minProperties"
  [context canonical-schema-path]
  (mk-check context canonical-schema-path
            (fn [min-properties]
              (fn [instance _ _]
                (when (map? instance)
                  (let [cnt (count (keys instance))]
                    (when (< cnt min-properties)
                      {:hints {:count cnt}})))))))

(defmethod key-validator "required"
  [context canonical-schema-path]
  (mk-check context canonical-schema-path
            (fn [required]
              (fn [instance _ _]
                (when (map? instance)
                  (when-let [missing (->> required
                                          (remove #(contains? instance %))
                                          seq)]
                    {:hints {:missing (vec missing)}}))))))

(defmethod key-validator "dependentRequired"
  [context canonical-schema-path]
  (mk-pred context canonical-schema-path
           (fn [dependent]
             (fn [instance]
               (if (map? instance)
                 (every? (fn [[k props]]
                           (if (contains? instance k)
                             (every? #(contains? instance %)
                                     props)
                             true))
                         dependent)
                 true)))))

(defmethod key-validator "enum"
  [context canonical-schema-path]
  (mk-pred context canonical-schema-path
           (fn [enum]
             (let [enum (set enum)]
               (fn [instance]
                 (contains? enum instance))))))

(defmethod key-validator "const"
  [context canonical-schema-path]
  (mk-pred context canonical-schema-path
           (fn [const]
             #(= const %))))

(defmethod key-validator "not"
  [{:keys [specification] :as context} canonical-schema-path]
  (let [canonical-schema-path (pointer/canonical-path specification canonical-schema-path)
        validator             (schema-validator context canonical-schema-path)]
    (fn [instance path schema-path]
      (when-not (validator instance path schema-path)
        [(keyword-error context
                        {:canonical-schema-path canonical-schema-path
                         :instance              instance
                         :path                  path
                         :schema-path           schema-path})]))))

;; TODO: prefixItems (???)
;; https://datatracker.ietf.org/doc/html/draft-bhutton-json-schema-01#section-10.3.1.1
;; TODO: additionalProperties, patternProperties (???)
;; https://datatracker.ietf.org/doc/html/draft-wright-json-schema-validation-01#section-6.20
;; TODO: if, then, else,
;; https://datatracker.ietf.org/doc/html/draft-bhutton-json-schema-01#section-10.2.2.1
;; TODO: dependentSchemas
