(ns nl.jomco.openapi.v3.path-matcher
  (:require
   [clojure.string :as string]))

;; given a set of openapi templated paths match a url.
;;
;; Prefer non-templated paths over templated paths, by ordering paths
;; by their non-templated parts (longest parts first).


(defn- tokenize-template
  [template]
  {:pre [(string? template)]}
  (when template
    (let [[prefix rst] (string/split template #"\{" 2)]
      (if rst
        (let [[param rst] (string/split rst #"\}" 2)
              tokens      (list {:tag :const :val prefix}
                                {:tag :param :val param})]
          (if (not= "" rst)
            (concat tokens (tokenize-template rst))
            tokens))
        (list {:tag :const :val prefix})))))

(defn- token-specifity
  [{:keys [tag val] :as token}]
  (cond
    (= :const tag)
    (count val)

    (= :param tag)
    -1

    (nil? token)
    0))

(defn- compare-token
  [x y]
  (- (compare (token-specifity x) (token-specifity y))))

(defn- compare-tokens
  [x y]
  (if (= x y) ;; also catch x = nil, y = nil
    0
    (let [r (compare-token (first x) (first y))]
      (if (zero? r)
        (recur (next x) (next y))
        r))))

(defn compare-templates
  [x y]
  (compare-tokens (tokenize-template x) (tokenize-template y)))

(defn- tokens-pattern
  [tokens]
  (->> tokens
       (map (fn [{:keys [tag val]}]
              (if (= :const tag)
                (java.util.regex.Pattern/quote val) ;; match literal string
                "([^/?#]*)")))
       (apply str)
       re-pattern))

(defn path-matcher
  "Return a matcher function that given a uri path, if the uri matches
  `template`, returns a map of :template and :parameters for the
  uri. Matcher returns `nil` if uri does not match."
  [template]
  (let [tokens          (tokenize-template template)
        path-parameters (filter (fn [{:keys [tag]}]
                                  (= tag :param)) tokens)
        regex (tokens-pattern tokens)]
    (fn [uri]
      (when-let [[_ & vals] (re-matches regex uri)]
        {:template   template
         :parameters (into {}
                           (map vector (map :val path-parameters) vals))}))))

(defn paths-matcher
  "Return a matcher given a collection of `templates`.

  When matchers is given a uri, for the most specific template
  matching uri, return a map of :template and :parameters. Returns
  `nil` when no template matches."
  [templates]
  (let [matchers (->> templates
                      (sort compare-templates) ;; sort by most specific
                      (map path-matcher))]
    (fn [uri]
      (some #(% uri) matchers))))
