(ns nl.jomco.ring-trace-context
  (:require [clojure.string :as string])
  (:import org.apache.commons.codec.binary.Hex
           java.util.Random))

(defn parse-traceparent
  [s]
  (when-let [[_ version] (re-find #"^([0-9a-f]{2})-" s)]
    (when (and (not= "ff" version)  ;; valid version
               (>= (count s) 55))   ;; valid header size
      (when-let [[_ trace-id parent-id flags]
                 (re-find #"^[0-9a-f]{2}-([0-9a-f]{32})-([0-9a-f]{16})-([0-9a-f]{2})" s)]
        (when (and (not= "00000000000000000000000000000000" trace-id)  ;; invalid trace-id
                   (not= "0000000000000000" parent-id)  ;; invalid parent-id
                   )
          (let [flag-bits (Long/parseLong flags 16)
                sampled?  (= 0x01 (bit-and flag-bits 0x01))]
            {:version     version
             :trace-id    trace-id
             :parent-id   parent-id
             :trace-flags (if sampled? #{:sampled} #{})}))))))

(defn generate-traceparent
  [{:keys [version trace-id parent-id] {:keys [sampled]} :trace-flags :or {version "00"}}]
  (str version "-" trace-id "-" parent-id "-" (if sampled "01" "00")))

(defn parse-tracestate
  [s]
  (let [list-members (-> (if (coll? s) (string/join "," s) s)
                         (string/trim)
                         (string/split #"[\t ]*,[\t ]*"))]
    (mapv #(string/split % #"=" 2) list-members)))

(defn- random-hex
  [num-bytes]
  (let [buff (byte-array num-bytes)]
    (.nextBytes (Random.) buff)
    (Hex/encodeHexString buff)))

(defn random-traceparent
  []
  {:version     "00"
   :trace-id    (random-hex 16)
   :parent-id   (random-hex 8)
   :trace-flags "00"})

(defn wrap-trace-context
  [f]
  (fn [request]
    (if-let [traceparent (-> request (get-in [:headers "traceparent"] parse-traceparent))]
      (f (assoc request
                :traceparent traceparent
                :tracestate (-> request (get-in [:headers "tracestate"] parse-tracestate))))
      (f (assoc request
                :traceparent (random-traceparent))))))
