(ns rill.message
  (:refer-clojure :exclude [type])
  (:require [schema.macros :as sm]
            [schema.core :as s]
            [rill.uuid :refer [new-id]]
            [rill.timestamp :refer [now]]
            [nl.zeekat.identifiers :refer [lisp-name]]))

(def id
  "The unique identifier of a message"
  ::id)

(def type
  "The type of the message"
  ::type)

(def number
  "The ordering number of the event in its original stream"
  ::number)

(def cursor
  "The ordering number of the event in the current stream
This may differ from message/number if the current stream is the all-event-stream"
  ::cursor)

(def timestamp
  "The creation time of the event"
  ::timestamp)

(def stream-id
  "The stream identifier for this message."
  ::stream-id)

(def partition-id
  "The identifier of the partition which this message belongs to."
  ::partition-id)

(defn ->type-keyword
  [ns sym]
  (keyword (name (ns-name ns)) (name sym)))

(defn params->args
  [params]
  (mapv #(symbol (name (first %)))
        (partition-all 2 params)))

(defmulti primary-aggregate-id
  "The id of the aggregate that will handle this message"
  type)
(defmulti aggregate-partition-id
  "The identifier of the partition for this message"
  type)

(defmulti observers
  "A collection of [id, handler-fn] pairs for each aggregate that should receive this message after it is successfully committed."
  type)

(defmethod observers
  :default
  [_]
  nil)

(defn make-message
  "Create a new message with type `message-type` and data"
  [message-type data]
  (assoc data
         type message-type
         id (new-id)
         timestamp (now)))

(defmacro commandmessage
  [type-keyword & params]
  {:pre [(every? keyword? (take-nth 2 (drop-last 2 params))) (even? (count params))]}
  (let [[primary-aggregate-id-fn partition-id-fn] (take-last 2 params)
        params (drop-last 2 params)
        args (params->args params)
        ks (map keyword args)]
    `(do
       (defmethod primary-aggregate-id ~type-keyword
         [message#]
         (~primary-aggregate-id-fn message#))
       (defmethod aggregate-partition-id ~type-keyword
         [message#]
         (~partition-id-fn message#))
       (fn ~(vec args)
         (let [msg# (make-message ~type-keyword ~(zipmap ks args))]
           (assoc msg# ~partition-id (~partition-id-fn msg#)))))))

(defmacro eventmessage
  [type-keyword & params]
  {:pre [(every? keyword? (take-nth 2 (drop-last params))) (odd? (count params))]}
  (let [primary-aggregate-id-fn (last params)
        params (drop-last params)
        args (params->args params)
        ks (map keyword args)]
    `(do
       (defmethod primary-aggregate-id ~type-keyword
         [message#]
         (~primary-aggregate-id-fn message#))
       (fn ~(vec args)
         (make-message ~type-keyword ~(zipmap ks args))))))

(defmacro defcommandmessage
  [name & params]
  (let [type-keyword (->type-keyword (ns-name *ns*) name)
        name-str (clojure.core/name name)]
    `(def
       ~(vary-meta (symbol (lisp-name name-str)) assoc
                   :doc (str "Create a new " name-str " command from the positional arguments. Automatically generates a new command id."))
       (commandmessage ~type-keyword ~@params))))

(defmacro defeventmessage
  [name & params]
  (let [type-keyword (->type-keyword (ns-name *ns*) name)
        name-str (clojure.core/name name)]
    `(def
       ~(vary-meta (symbol (lisp-name name-str)) assoc
                   :doc (str "Create a new " name-str " event from the positional arguments. Automatically generates a new event id."))
       (eventmessage ~type-keyword ~@params))))

(defmacro command [& args] `(commandmessage ~@args))
(defmacro event [& args] `(eventmessage ~@args))
(defmacro defcommand
  [name & params]
  `(defcommandmessage ~name ~@params))
(defmacro defevent
  [name & params]
  `(defeventmessage ~name ~@params))