(ns joints.boot.task
  (:require
   [boot.core :as bc]
   [boot.util :as bu]
   [clojure.tools.namespace.dir :as nsd]
   [clojure.tools.namespace.track :as nst]
   [joints.boot.repl :as br]))

(defn restart-system?
  [system-symbol prev-fileset next-fileset files {:keys [paths regexes]}]
  (when (some? system-symbol)
    (if (empty? files)
      (bu/info (str "No `files` to be watched. "
                    "Will not attempt to manage `system`'s lifecycles.\n"))
      (let [input-files (->> next-fileset
                             (bc/fileset-diff prev-fileset)
                             bc/input-files)
            query-fn    (cond
                          paths
                          bc/by-path

                          regexes
                          bc/by-re

                          :else
                          bc/by-name)
            files (if (some? regexes) (map re-pattern files) files)]
        (->> input-files (query-fn files) seq boolean)))))

(defn validate-option
  [{:keys [paths regexes]}]
  (when (and (true? paths) (true? regexes))
    (bu/fail "Cannot specify both `paths` and `regexes` to `true`")
    (throw (ex-info "Incorrect option" {}))))

(bc/deftask system
  [s system VAL  sym   "Symbol of the system. Should point to a nullary fn"
   f files  VALS [str] "Files to be watched to restart the system. Name-based."
   p paths       bool  "`files` as path. Pick between this and `regexes`"
   r regexes     bool  "`files` as regex. Pick between this and `paths`"]
  (validate-option *opts*)
  (let [prev-fileset_ (atom nil)
        dirs          (bc/get-env :directories)
        tracker_      (atom (nsd/scan-dirs (nst/tracker) dirs))
        init-system   (if (some? system)
                        (delay
                         (bu/info "Starting system...\n")
                         (br/setup! system)
                         (br/boot!))
                        (delay
                         (bu/info (str "No system is supplied. "
                                       "Will only refresh namespaces.\n"))))]
    (fn [next-task]
      (fn [fileset]
        (when (realized? init-system)
          (swap! tracker_ nsd/scan-dirs)
          (br/reboot! @tracker_ (restart-system? system
                                                 @prev-fileset_
                                                 fileset
                                                 files
                                                 {:paths   paths
                                                  :regexes regexes})))
        @init-system
        (next-task (reset! prev-fileset_ fileset))))))
