(ns joints.components.bidi.ring
  (:require
   [bidi.bidi :as b]
   [bidi.ring :refer [make-handler]]
   [com.stuartsierra.component :as c]
   [joints.components.bidi.proto :as cbp]
   [joints.components.ring.core :as crc]
   [ring.util.http-response :as resp]))

(defn search-route-config
  [{:keys [prefix] :as m}]
  (when-let [collected (->> m
                            (map val)
                            (filter (comp
                                     #(satisfies? cbp/IResourceProvider %)
                                     #(satisfies? b/RouteProvider %)))
                            not-empty)]
    [[(or prefix "") (map b/routes collected)]
     (map cbp/resources collected)]))

(defn construct-handler
  [{:keys [not-found-handler default-handler]} route-config]
  (if (some? route-config)
    (some-fn
     (apply make-handler route-config)
     (or not-found-handler
         (constantly
          (resp/not-found "No matching path found"))))
    (or default-handler
        (constantly
         (resp/service-unavailable
          "No route config found")))))

(defrecord RingRouter [prefix routes not-found-handler default-handler handler]
  c/Lifecycle
  (start [this]
    (if (some? handler)
      this
      (let [[routes :as route-config]
            (search-route-config this)

            handler (construct-handler this route-config)]
        (assoc this :routes routes :handler handler))))
  (stop [this]
    (if (nil? handler)
      this
      (assoc this :routes nil :handler nil)))

  b/RouteProvider
  (routes [_]
    routes)

  crc/IRequestHandler
  (-request-handler [_]
    handler))

(defn make-ring-router
  ([{:keys [prefix not-found-handler default-handler]}]
   (map->RingRouter {:prefix            prefix
                     :not-found-handler not-found-handler
                     :default-handler   default-handler}))
  ([]
   (make-ring-router {})))

(defrecord RingEndpoint [routes resources]
  b/RouteProvider
  (routes [_]
    routes)

  cbp/IResourceProvider
  (-resources [_]
    resources))

(defn make-ring-endpoint
  [{:keys [routes resources]}]
  (map->RingEndpoint {:routes    routes
                      :resources resources}))
