(ns pos5.app
  (:require
   [clojure.spec.alpha :as s]
   [buddy.core.keys :as bdyks]
   [buddy.sign.jwt :as bdysgnjwt]
   [pos5.request :as req]
   [pos5.url :as url]))

;; -------------------------------------------------------------------
;; app client spec
;; -------------------------------------------------------------------

(s/def ::host-url
  string?)

(s/def ::content-type
  #{:transit :json})

(s/def ::sha-algorithm
  #{:hs256 :hs512})

(s/def ::sha-secret
  string?)

(s/def ::asymetric-algorithm
  #{:rs256 :rs512 :ps256 :ps512})

(s/def ::public-key-path
  (s/or :string string?
        :url    #(instance? java.net.URL %)))

;; -------------------------------------------------------------------
;; protocols
;; -------------------------------------------------------------------

(defprotocol IAppClient
  (-reset-token [this username])
  (-reset-password [this token username new-password]))

;; -------------------------------------------------------------------
;; app client
;; -------------------------------------------------------------------

(defn- forget-token
  [algorithm secret username]
  (bdysgnjwt/sign {:username username} secret {:alg algorithm}))

(defrecord AppClient [host-url content-type algorithm secret]
  IAppClient
  (-reset-token [_ username]
    (req/POST (url/request-reset-token-url host-url username)
              (-> {}
                  (req/wrap-content-type content-type)
                  (req/wrap-forget-token
                   (forget-token algorithm secret username)))))
  (-reset-password [_ token username new-password]
    (req/PUT (url/target-user-url host-url username)
             (-> {}
                 (req/wrap-content-type content-type)
                 (assoc :params {:password new-password})
                 (req/wrap-reset-token token)))))

(defn make-sha-app-client
  [host-url content-type algorithm secret]
  (->AppClient (s/assert ::host-url host-url)
               (s/assert ::content-type content-type)
               (s/assert ::sha-algorithm algorithm)
               (s/assert ::sha-secret secret)))

(defn make-asymetric-app-client
  [host-url content-type algorithm public-key-path]
  (let [public-key (bdyks/public-key public-key-path)]
    (->AppClient (s/assert ::host-url host-url)
                 (s/assert ::content-type content-type)
                 (s/assert ::asymetric-algorithm algorithm)
                 (s/assert ::public-key-path public-key))))

(defn reset-token
  [iapp-client username]
  (-reset-token iapp-client username))

(defn reset-password
  [iapp-client token username new-password]
  (-reset-password iapp-client token username new-password))
