(ns m.hi
  (:require-macros [cljs.core.async.macros :refer [go go-loop]])
  (:require [cljs.core.async :refer [put! chan]]
            [reagent.core :as r]
            [re-frame.core :refer [reg-sub subscribe]]
            [m.core :refer [e m]]
            [m.client :refer [s a get-buffer]]
            [m.xfs :refer [serialize deserialize files-to-chan ism?]]
            [m.async :refer [*>]]
            [m.peer :refer [create-peer connect-peer send-data destroy-peer stun-config]]))

(defonce hi-div-id "hi-div")

(reg-sub
  :peer/peers
  (fn [db] (:peer/peers db)))

(reg-sub
  :hi/peers
  :<- [:peer/peers]
  (fn [peers _]
    (remove #(nil? (:peer (val %))) peers)))

(reg-sub
  :hi/stream
  (fn [db] (:hi/stream db)))

(defn video-stream [id stream]
  (js/console.log "video-stream")
  (let [el (.createElement js/document "video")]
    (set! (.-autoplay el) true)
    (set! (.-width el) "300")
    (set! (.-srcObject el) stream)
    el))

(defn show-peer-stream [id stream]
  (let [parent (.getElementById js/document (name id))
        child (video-stream id stream)]
    (.insertBefore parent child (.-firstChild parent))
    (m {:peer/peers {id {:hi/visible true}}})))

(defn send-chat [id event]
  (when (= "Enter" (.-key event))
    (send-data id {:chat (.. event -target -value)})
    (set! (.. event -target -value) "")))

(defn hi-ui []
  (let [v (r/atom nil)]
    (fn []
      (let [peers @(subscribe [:hi/peers])
            media @(subscribe [:hi/stream])
            config (merge {:trickle false :stream media} stun-config)]
        [:div
         [:input
          {:placeholder "id"
           :value @v
           :on-change #(reset! v (.. % -target -value))}]
         [:button
          {:on-click #(create-peer (keyword @v) (merge {:initiator true} config))
           :disabled (empty? @v)}
          "hi?"]
         [:button
          {:on-click #(connect-peer @v config)
           :disabled (empty? @v)}
          "hi!"]
         (for [[id {:keys [stream hi/visible]}] peers]
           ^{:key id}
           [:div {:id id}
            (when (and stream (not visible))
              [:button {:on-click #(show-peer-stream id stream)} "showmewhatyougot"])
            [:input
             {:placeholder "chat"
              :on-key-down #(send-chat id %)}]
            [:button {:on-click #(destroy-peer id)} "remove you"]])]))))

(defn handle-media [div media]
  (.appendChild div (video-stream :hi-me media))
  (m {:hi/stream media}))

(defn hi [& [constraints]]
  (let [app (.getElementById js/document "app")
        div (.createElement js/document "div")]
    (set! (.-id div) hi-div-id)
    (if-let [old (.getElementById js/document hi-div-id)]
      (do (.replaceChild app div old)
          (for [[k _] (e :peer/peers)]
            (m {:peer/peers {k {:hi/visible false}}})))
      (.appendChild app div))
    (r/render [hi-ui] div)
    (when constraints
      (-> (.getUserMedia (.-mediaDevices js/navigator) (clj->js constraints))
          (.then #(handle-media div %))))))

(defn bi []
  (js/console.log "bi")
  (when-let [local-stream (e :hi/stream)]
    (map #(.stop %) (.getTracks local-stream))
    (m {:hi/stream nil}))
  (for [[k _] (e :peer/peers)]
    (m {:peer/peers {k {:hi/visible false}}}))
  (.remove (.getElementById js/document hi-div-id)))
