(ns obis-shared.config
  "Expects to find a file called obis-shared.properties
  on the classpath with properties ."
  (:import java.util.Properties)
  (:require [clojure.java.io :as jio]
            [clojure.walk :as wk])
  (:use [obis-shared.utils.gen-class]))

(def ^:private props nil)

(defn stringy-map
  "Stringify keys, but better, except not recursive."
  [m]
  (let [ks (keys m)]
    (zipmap (map #(if (instance? clojure.lang.Named %) (name %) (str %)) ks)
            (map #(get m %) ks))))

(defn set-properties
  "Argument should be a map e.g.
    {\"obis.entity.host\" \"http://foo.bar\",
     \"obis.entity.port\" 4844,
     \"activiti.explorer.url\" \"http://foo.bar/\"}"
  [m]
  (alter-var-root (var props) (constantly (stringy-map m)))
  true)

; There's probably a cleaner way to do this.
(defn- find-props-file
  []
  (first
    (keep jio/resource
          "obis-shared.properties"
          "config/obis-shared.properties"
          "lib/obis-shared.properties")))

(def ^:private config-file-props
  (delay
    (when-let [res (jio/resource "obis-shared.properties")]
      (doto (new Properties)
            (.load (jio/reader res))))))

(defn get-prop
  [prop-name]
  (or (get props prop-name)
      (when @config-file-props
        (.get @config-file-props prop-name))
      (throw (new Exception
                  (str "Cannot find property value for "
                       prop-name
                       ". Try adding an \"obis-shared.properties\" "
                       "file on the classpath or calling "
                       "obis-shared.config.set-properties with a map.")))))

(gen-class-with-exports)
