(ns obis-shared.entity.core
  (:use obis-shared.utils.gen-class)
  (:use [clojure.data.json :only (json-str write-json read-json)])
  (:require [obis-shared.entity :as e]
            [obis-shared.entity.identity :as i]
            [obis-shared.entity.services :as s]
            [obis-shared.entity.organizations :as o]
            [obis-shared.entity.notifications :as n]
            [obis-shared.entity.service-requests :as sr]
            [clojure.walk :as walk]
            [obis-shared.entity.utils :as u]))

(def ^:dynamic services)
(def ^:dynamic identities)
(def ^:dynamic projects)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; SERVICE REQUEST FUNCTIONS FOR ADMIN
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defn filter-by-id
  [coll id]
  (first (doall (filter #(= id (:id %)) coll))))

(defn get-service
  [id]
  (filter-by-id services id))

(defn get-identity
  [id]
  (filter-by-id identities id))

(defn get-project
  [id]
  (filter-by-id projects id))

(defn project-role-to-identity
  [role-relationship]
  (let [identity-id (:to role-relationship)
        i (get-identity identity-id)
        r (get-in role-relationship [:attributes :role])]
    [r i]))

(defn all-project-roles-and-identities
  [obis-id]
  (into {} (doall
            (map project-role-to-identity (e/all-project-roles obis-id)))))

(defn- owner-of-ssr
  [ssr-id sr-id]
  (let [ssr-owners (sr/relationships-by-type sr-id
                                             i/sub-service-request-owner-relationship-type)
        ssr-owner (first ssr-owners)]
    (if ssr-owner
      (get-identity (:to ssr-owner)))))

;;TODO: Pass org-map into this function.
(defn organization-tree-for-service-provider
  [identity-id org-map]
  (let [top-level-sp-orgs (i/service-provider-for-top-level-organizations identity-id)]
    (flatten
     (doall (map #(o/all-related-organizations org-map %) top-level-sp-orgs)))))

(defn service-request-has-relationships
  [sr-id]
  (not= [] (sr/relationships sr-id)))

(defn filter-out-orphan-service-requests
  [service-requests]
  (doall
   (filter #(service-request-has-relationships (:id %)) service-requests)))

(defn service-requests-for-service-provider
  "Retrieves all the Service Requests for a Service Provider.
   - identity-id is the OBISID of the Service Provider
   - org-map is the Organizational Map"
  [identity-id org-map]
  (-> identity-id
      (organization-tree-for-service-provider org-map)
      (sr/ssrs-for-organizations)
      (filter-out-orphan-service-requests)))

(defn services-for-sub-service-request
  [ssr-id sr]
  (doall
   (map get-service
        (sr/service-ids-for-sub-service-request ssr-id
                                                sr))))

(defn related-organizations-to-sub-service-request
  [ssr-id sr]
  (let [srvcs (services-for-sub-service-request ssr-id sr)]
    (doall
     (map s/related-organizational-unit srvcs))))

(defn get-role-name
  [role-type roles]
  (let [role (get roles role-type)]
    (str (:first_name role) " " (:last_name role))))

(defn format-sr-data-for-admin-portal [sr-data]
  (dissoc (merge sr-data {:pi (get-role-name "pi" (:roles sr-data))})
          :roles))

(defn status-for-sub-service-request
  [ssr-id sr]
  (let [sr-status (get-in sr [:attributes :status] )
        sr-status-date (get-in sr [:attributes :submitted_at] )
        ssr (sr/get-sub-service-request ssr-id sr)
        ssr-status (:status ssr)
        ssr-status-date (:status_date ssr)]
    (if (and ssr-status ssr-status-date)
      [ssr-status ssr-status-date]
      [sr-status sr-status-date])))

(defn normalize-sub-service-request
  [[ssr-id sr]]
  (let [sr-id (:_id sr)
        status-status-date (status-for-sub-service-request ssr-id sr)
        status (first status-status-date)
        status-date (last status-status-date)
        prj-id (sr/related-project sr-id)  ;; no opportunity
        project (get-project prj-id) 
        short-title (get project :short_title)
        services (doall (map :name (services-for-sub-service-request ssr-id sr)))
        project-roles (all-project-roles-and-identities prj-id)
        service-requester (get-identity (get-in sr [:attributes :service_requester_id]))
        owner (owner-of-ssr ssr-id sr-id)
        friendly-id (get-in sr [:attributes :friendly_id])
        project-friendly-id (:friendly_id project)]
    {:sr_id sr-id
     :ssr_id ssr-id
     :display_id (str project-friendly-id "-" ssr-id)
     :status status
     :status_date status-date
     :project_id prj-id
     :short_title short-title
     :services services
     :roles project-roles
     :friendly_id friendly-id
     :service_requester (str (:first_name service-requester) " " (:last_name service-requester))
     :person_assigned (str (:first_name owner) " " (:last_name owner))}))

(defn all-service-requests-for-service-provider
  "Returns all of the Service Requests related to a Service Provider.
     - identity-id is the obis-id of the Service Provider.
     - org-map is the organizational map
     - services is a list of Services
     - identities is a list of all the Identities
     - projects is a list of Projects"
  [identity-id org-map services identities projects]
  (binding [services services
            identities identities
            projects projects]
    (doall (map #(format-sr-data-for-admin-portal
                  (normalize-sub-service-request (:value %)))
                (service-requests-for-service-provider identity-id org-map)))))

(defn all-srs-for-sp
  [identity-id]
  (binding [services (s/all)
            identities (i/all)
            projects (e/all-projects)]
    (let [org-map (o/build-organization-map)]
      (doall (map #(format-sr-data-for-admin-portal (normalize-sub-service-request (:value %)))
                  (service-requests-for-service-provider identity-id org-map))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; UPDATE SERVICE REQUESTS 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defn any-ctrc-related-services
  [org-map ssr-id sr]
  (let [ctrc-and-descendants (o/ctrc-and-descendants org-map)
        related-orgs (related-organizations-to-sub-service-request ssr-id sr)]
    (not (nil? (some ctrc-and-descendants related-orgs)))))

(defn complete-sub-service-request
  [service-request sub-service-request-id]
  (do
    (sr/modify-sub-service-request-status service-request
                                          sub-service-request-id
                                          "complete")))

(defn identities-to-notify
  [roles]
  (doall
   (filter #(get-in % [:attributes :receive_notification]) roles)))

(defn send-status-notification
  [role]
  (let [obis-id (:to role)
        identity (i/get-identity obis-id)]
    (n/create-notification "status"
                           (:uid identity)
                           (:email identity)
                           "Status Change"
                           "Test. Ignore email.")))

(defn send-status-notifications
  [to-notify]
  (doall (map #(send-status-notification %) to-notify)))

;;(defmulti update-service-request-status (fn [ssr-id sr-id status] status))
 
;;TODO: Update to capture the status-date.
;;TODO: FIX THE METHODS TO ENABLE THREADING. UGH. THIS IS SO UGLY.
(defn update-sub-service-request-status
  [ssr-id sr-id status]
  (let [p-id (sr/related-project sr-id)
        roles (e/all-project-roles p-id)
        to-notify (identities-to-notify roles)
        service-request (sr/get-service-request sr-id)
        service-request (sr/modify-sub-service-request-past-status-list ssr-id
                                                                        service-request)
        service-request (sr/modify-sub-service-request-status ssr-id
                                                              service-request
                                                              status)
        service-request (sr/modify-sub-service-request ssr-id
                                                       service-request
                                                       "status_date"
                                                       (.format u/yyyy-mm-dd (java.util.Date.)))]
    (do
      ;;(println (str "atts: " (:attributes service-request)))
      ;;(println (str "identifiers: " (:identifiers service-request)))
      (sr/update-service-request (sr/obis-id service-request)
                                 (:attributes service-request)
                                 (:identifiers service-request))
      (send-status-notifications to-notify))))


(defn assign-owner-to-sub-service-request
  [identity-id ssr-id sr-id ]
  (let [ssr-owners (sr/relationships-by-type sr-id 
                                             i/sub-service-request-owner-relationship-type)]
    (do
      (doall
       (map i/delete-relationship ssr-owners))
      (i/make-sub-service-request-owner identity-id
                                        sr-id
                                        ssr-id))))

;; (defn mark-service-request-complete-if-appropriate
;;   [service-request-id]
;;   (let [sr (sr/get-service-request service-request-id)]
;;     (if (sr/all-ssr-statuses-complete sr)
;;       (complete-service-request service-request-id)))

;;BUSINESS RULES
;; If all sub-service-requests are DRAFT or SUBMITTED, then the options for a
;; USER are VIEW and EDIT.

;; If any sub-service-request is in a status other than DRAFT or
;; SUBMITTED, then the options are VIEW only.
(defn nil-or-submitted-or-draft
  [status]
  (or (= "submitted" status)
      (= nil status)
      (= "draft" status)))

(defn distinct-ssr-statuses
  [sr]
  (set (vals (sr/sub-service-request-statuses sr))))

(defn can-service-request-be-edited
  [sr]
  (every? nil-or-submitted-or-draft (distinct-ssr-statuses sr)))

(defn all-ssr-statuses-complete
  [sr]
  (every? #{"complete"} (distinct-ssr-statuses sr)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; STATUSES
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Need a function indicating candidate statuses.
(defn is-draft-a-candidate-status
  [sr]
  (if (can-service-request-be-edited sr)
    true
    false))

(defn is-submitted-a-candidate-status
  [sr]
  false)

(defn is-in-process-a-candidate-status
  [sr]
  true)

(defn is-ctrc-review-a-candidate-status
  [org-map ssr-id sr]
  (any-ctrc-related-services org-map ssr-id sr))

(defn is-ctrc-approved-a-candidate-status
  [org-map ssr-id sr]
  (any-ctrc-related-services org-map ssr-id sr))

(defn is-complete-a-candidate-status
  [sr]
  true)

(defn is-awaiting-pi-approval-a-candidate-status
  [sr]
  true)

(defn is-on-hold-a-candidate-status
  [sr]
  true)

(defn candidate-statuses-for-service-request
  [org-map ssr-id sr-id]
  (let [candidate-statuses (atom [])
        sr (sr/get-service-request sr-id)]
    (if (is-draft-a-candidate-status sr)
      (swap! candidate-statuses conj "draft"))
    (if (is-submitted-a-candidate-status sr)
      (swap! candidate-statuses conj "submitted"))
    (if (is-in-process-a-candidate-status sr)
      (swap! candidate-statuses conj "in process"))
    (if (is-complete-a-candidate-status sr)
      (swap! candidate-statuses conj "complete"))
    (if (is-ctrc-review-a-candidate-status org-map ssr-id sr)
      (swap! candidate-statuses conj "ctrc review"))
    (if (is-ctrc-approved-a-candidate-status org-map ssr-id sr)
      (swap! candidate-statuses conj "ctrc approved"))
    (if (is-awaiting-pi-approval-a-candidate-status sr) 
      (swap! candidate-statuses conj "awaiting pi approval"))
    (if (is-on-hold-a-candidate-status sr) 
      (swap! candidate-statuses conj "on hold"))
    @candidate-statuses))

(defn candidate-statuses-for-ssr
  [ssr-id sr-id]
  (binding [services (s/all)
            ;;identities (i/all)
            ;;projects (e/all-projects)
            ]
    (let [org-map (o/build-organization-map)]
      (candidate-statuses-for-service-request org-map ssr-id sr-id))))

;; PRIORITY B
;; Need an archived function for sub-service-requests.
;; - Add to the list of statuses
;; - Archived sub-service-requests are not brought back in the main
;; return function but as a separate call i.e. they are lazy loaded.

;; Super Admin has access to Service Requests in DRAFT status.
;; - view_draft

;; Search - lower priority

;; It has to be in CTRC Review for it to be CTRC Approved.

;; Notifications for In Process, CTRC Review, CTRC Approved
;; - 
;; Emails for Submitted and Completed

(defn sub-service-request-status
  [ssr-id sr-id]
  (let [sr (sr/get-service-request sr-id)
        sr-status (sr/service-request-status sr)
        ssr-status (first (sr/sub-service-request-status-and-status-date ssr-id sr))]
    (if ssr-status
      ssr-status
      sr-status)))

(defn organization-parent-id-key
  [org-node]
  (some #{:program_id :core_id :provider_id} (keys org-node)))

(defn organization-parent-id
  [org-id]
  (let [org (o/organization org-id)
        k (organization-parent-id-key org)]
    (get org k)))

(defn organizational-ancestors
  [org-id]
  (loop [id org-id
         result []]
    (if id
      (recur (organization-parent-id id) 
             (conj result id))
      result)))

(defn service-providers-for-organizational-unit
  [org-id]
  (doall
   (map :from (o/relationships-by-type org-id i/service-provider-relationship-type))))

(defn ancestors-of-related-organizations
  [related-organizations]
  (doall
   (flatten (map organizational-ancestors related-organizations))))

(defn potential-service-provider-ids-for-ssr
  [ssr-id sr-id]
  (binding [obis-shared.entity.core/services (s/all)]
    (let [sr (sr/get-service-request sr-id)
          related-organizations (related-organizations-to-sub-service-request ssr-id sr)
          all-organizations (ancestors-of-related-organizations related-organizations)]
      (doall
       (flatten (map #(service-providers-for-organizational-unit %) all-organizations))))))

(defn potential-service-providers-for-ssr
  [ssr-id sr-id]
  (doall
   (map i/get-identity (potential-service-provider-ids-for-ssr ssr-id
                                                               sr-id))))


(defn ssr-owner
  [ssr-id sr-id]
  (let [ssr-owners (sr/relationships-by-type sr-id
                                             i/sub-service-request-owner-relationship-type)
        ssr-owner (first ssr-owners)]
    (if ssr-owner
      (i/get-identity (:to ssr-owner)))))

(defn ssr-past-statuses
  [ssr-id sr-id]
  (let [sr (sr/get-service-request sr-id)]
    (sr/sub-service-request-past-status-list ssr-id sr)))

(gen-class-with-exports)
