(ns obis-shared.entity.identity
  (:use obis-shared.utils.gen-class
        obis-shared.entity.utils
        clojure.set)
  (:require [obis-shared.entity.http-json :as http]))

(def service-provider-relationship-type "service_provider_organizational_unit")
(def sub-service-request-owner-relationship-type "service_request_owner")

(defn all
  "Return all identities."
  []
  (http/get-simple ["identities"]))

(defn create-identity
  ([attrs] (create-identity attrs {}))
  ([attrs identifiers]
     (http/post ["identities"] 
        {:attributes attrs,
         :identifiers identifiers})))

(defn get-identity
  [id]
  (http/get-simple ["identities" id]))

(defn get-identity-by-ldap
  [ldap-uid]
  (http/get ["identifiers" "ldap_uid" ldap-uid]))

(defn delete-identity
  [obisid]
  (http/delete ["identities" obisid]))

(defn identities-for-project
  [proj-id]
  (http/get-simple ["projects" proj-id "identities"]))

(defn identities-for-service-request
  [sr-id]
  (http/get-simple ["service_requests" sr-id "identities"]))

(defn projects-for-identity
  [identity-id]
  (http/get-simple ["identities" identity-id "projects"]))

(defn relationships
  [identity-id]
  (http/get ["identities" identity-id "relationships"]))

;;TODO: MEMOIZE
(defn relationships-by-type
  [identity-id relationship-type]
  (let [r (relationships identity-id)]
    (filter #(= (:relationship_type %) relationship-type) r)))

(defn service-provider-relationships
  [identity-id]
  (relationships-by-type identity-id service-provider-relationship-type))

(defn relate-to
  [relationship from-id to-id attributes]
  (http/post ["identities" from-id "relationships"]
             (new-relationship relationship
                               from-id
                               to-id
                               attributes)))

;;DELETE /OBISENTITY/:PCLASS/:OBISID/RELATIONSHIPS/:RELATIONSHIP_ID
(defn delete-relationship
  [relationship]
  (let [identity-id (:to relationship)
        relationship-id (:relationship_id relationship)]
    (http/delete ["identities" identity-id "relationships" relationship-id])))

(defn make-service-provider
  [identity-id service-provider-id]
  (relate-to service-provider-relationship-type
             identity-id
             service-provider-id
             {}))

(defn make-sub-service-request-owner
  [identity-id service-request-id sub-service-request-id]
  (relate-to sub-service-request-owner-relationship-type
             service-request-id             
             identity-id
             {:sub-service-request-id sub-service-request-id}))

(defn service-provider-for-top-level-organizations
  [identity-id]
  (set
   (doall (map :to (service-provider-relationships identity-id)))))

(gen-class-with-exports)

