(ns obis-shared.jruby
  "Functions for converting clojure data structures
  to Ruby data structures."
  (:import clojure.lang.Reflector))

; Should make this changeable by a method that is itself
; exported with the gen-class util stuff, so setting it
; from Ruby is easy.
;
; Also consider to defaulting to true, but only do it if
; JRuby is on the classpath? Convention over configuration!
(def return-jruby false)

(defmacro construct
  [class-name & args]
  `(Reflector/invokeConstructor
     (Class/forName ~(str class-name))
     (to-array [~@args])))

(defmacro invoke-static
  [class-name method-name & args]
  `(Reflector/invokeStaticMethod
     ~(str class-name)
     ~(str method-name)
     (to-array [~@args])))

(let [res? (delay (try
                    (Class/forName "org.jruby.Ruby")
                    true
                    (catch ClassNotFoundException c false)))]
  (def jruby? (partial deref res?)))

(defn ruby-instance
  []
  (or (invoke-static org.jruby.Ruby
                     getThreadLocalRuntime)
      (invoke-static org.jruby.Ruby
                     getGlobalRuntime)
      (throw (new Exception "Cannot find instance of org.jruby.Ruby!"))))

(defn ruby-nil
  []
  (construct org.jruby.RubyNil (ruby-instance)))

(defmulti to-jruby type)

(defmethod to-jruby clojure.lang.Sequential
  [coll]
  (invoke-static org.jruby.RubyArray
                 newArray
                 (ruby-instance)
                 (map to-jruby coll)))

(defmethod to-jruby Long
  [n]
  (invoke-static org.jruby.RubyFixnum
                 newFixnum
                 (ruby-instance)
                 n))

(defmethod to-jruby clojure.lang.IPersistentMap
  [m]
  (invoke-static org.jruby.RubyHash
                 newHash
                 (ruby-instance)
                 (zipmap
                   (map to-jruby (keys m))
                   (map to-jruby (vals m)))
                 (ruby-nil)))

;; (defmethod to-jruby clojure.lang.Keyword
;;   [kw]
;;   (invoke-static org.jruby.RubySymbol
;;                  newSymbol
;;                  (ruby-instance)
;;                  (name kw)))

(defmethod to-jruby clojure.lang.Keyword
  [s]
  (invoke-static org.jruby.RubyString
                 newString
                 (ruby-instance)
                 (name s)))

(defmethod to-jruby String
  [s]
  (invoke-static org.jruby.RubyString
                 newString
                 (ruby-instance)
                 s))

(defmethod to-jruby Boolean
  [b]
  (invoke-static org.jruby.RubyBoolean
                 newBoolean
                 (ruby-instance)
                 b))

(defmethod to-jruby nil
  [_]
  (construct org.jruby.RubyNil (ruby-instance)))

(defmethod to-jruby :default
  [x]
  (invoke-static org.jruby.javasupport.JavaObject
                 wrap
                 (ruby-instance)
                 x))
