(ns strum.core
  (:require [clojure.string :as string]))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Helper Functions

(defn- take-if
  ([pred coll] (take-if pred coll nil))
  ([pred [head & tail :as coll] default]
     (if (pred head) [head tail][default coll])))

(defn- positive-index [index coll]
  (if (>= index 0)
    index
    ;; str length - (index % str length)
    (- (count coll) (mod (Math/abs index) (count coll)))))

(defn- closing-bracket-index
  [string index]
  (loop [charlist (vec string)
         stack [\{] ;; begin by loading open bracket onto stack
         i (+ 2 index)]
    (if (empty? stack)
      i ;; base case is that brackets  have balanced
      (case (nth charlist i)
        \{ (recur charlist (conj stack char) (inc i))
        \} (recur charlist (pop stack) (inc i))
        (recur charlist stack (inc i))))))

(defn- extract-tokens [s]
  (loop [index 0
         tokens []]
    (let [next-index (.indexOf s "#{" index)]
      (if (= -1 next-index)
        tokens
        (let [end-index (closing-bracket-index s next-index)
              token (.substring s next-index end-index)]
          (recur end-index (conj tokens token)))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Public Functions

;; Functions Shadowed from clojure.string
(def split #(when % (string/split % %2)))
(def reverse #(when % (string/reverse %)))
(def replace #(when % (apply string/replace % %&)))
(def replace-first #(when % (apply string/replace-first % %&)))
(def trim-newline #(when % (string/trim-newline %)))
(def triml #(when % (string/triml %)))
(def trimr #(when % (string/trimr %)))
(def trim #(when % (string/trim %)))
(def upper-case #(when % (string/upper-case %)))
(def lower-case #(when % (string/lower-case %)))
(def escape #(when % (string/escape % %2)))
(def split-lines #(when % (string/split-lines %)))
(def join string/join)
(def blank? string/blank?)

(defn prefix?
  "Returns true if the String starts with the given prefix

   Examples:

     (prefix? \"Hello\" \"Hell\") => true

     (prefix? \"Hello\" \"Bye\") => false"
  [^String s ^String prefix]
  (assert (string? prefix) "Prefix should be a String")
  (if s (.startsWith s prefix) false))

(defn suffix?
  "Returns true if the String ends with the given suffix

   Examples:

     (suffix? \"Hello\" \"llo\") => true

     (suffix? \"Hello\" \"Bye\") => false"
  [^String s ^String suffix]
  (assert (string? suffix) "Suffix should be a String")
  (if s (.endsWith s suffix) false))

(defn index
  "Returns the index of the first occurance of the given pattern in the string.
   Returns nil if not found.

   Examples:

     (index \"Hello Hello\" \"ell\") => 1

     (index \"Hello Hello\" \"ell\" 4) => 6

     (index nil \"str\") => nil"
  ([^String s ^String pattern] (index s pattern 0))
  ([^String s ^String pattern offset]
     (assert (string? pattern) "Pattern should be a String")
     (assert (integer? offset) "Offset should be an Integer")
     (-> (if s (.indexOf s pattern offset))
         (#(if (= -1 %) nil %)))))

(defn rindex
  "Returns the index of the last occurance of the given pattern in the string.
   Returns nil if not found.

   Examples:

     (rindex \"Hello Hello\" \"ell\") => 6

     (rindex \"Hello Hello\" \"ell\" 4) => 1

     (rindex nil \"str\") => nil"
  ([^String s ^String pattern] (rindex s pattern (count s)))
  ([^String s ^String pattern offset]
     (assert (string? pattern) "Pattern should be a String")
     (assert (integer? offset) "Offset should be an Integer")
     (-> (if s (.lastIndexOf s pattern offset))
         (#(if (= -1 %) nil %)))))

(defn contains?
  "Returns true if the string contains the given substring.

   Examples:

     (contains? \"Spider-man, Spider-man\" \"der\") => true

     (contains? \"To the batcave\" \"base\") => false"
  [^String s ^String substring]
  (assert (string? substring) "Substring should be a String")
  (if s (.contains s substring) false))

(defn slice
  "Simple String slicing blatently and unashamedly taken from Python

   Examples:

     (slice \"123456\" 2 :- 4) => \"34\"

     (slice \"123456\" 2 :- ) => \"3456\"

     (slice \"123456\" :- 4) => \"1234\"

     (slice \"123456\" :- ) => \"123456\"

     (slice \"123456\" -4 :- -2) => \"34\""
  [^String s & args]
  (assert (clojure.core/contains? (set args) :-) "Expected keyword :-")
  (let [[lower args] (take-if integer? args)
        [sep args] (take-if keyword? args)
        [upper args] (take-if integer? args)
        lower (positive-index (or lower 0) s)
        upper (positive-index (or upper (count s)) s)]
    (if s (if (> upper lower) (.substring s lower upper) ""))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Public Macros

(defmacro fmt
  "Provides Ruby-like String interpolation

   Examples:

     (def name \"Batman\")

     (fmt \"Bruce Wayne is #{name}\") => \"Bruce Wayne is Batman\"

     (fmt \"The answer is #{(* 6 (#{6 7 8} 7))}\") => \"The answer is 42\""
  [^String s]
  (let [tokens (extract-tokens s)
        fstr (reduce #(.replace % %2 "%s") s tokens)
        fargs (->> (map #(slice % 2 :- ) tokens)
                   (map read-string))]
    `(->> (map #(or % "") [~@fargs]) ;; replace nils with empty strings
          (apply format ~fstr))))
