(ns backend-adapters.dynamodb.perform
  (:require [cljs.core.async :as async]
            [shared.protocols.loggable :as log]
            [shared.models.payload.index :as payload]
            [shared.protocols.convertible :as cv]
            [shared.protocols.specced :as sp])
  (:require-macros [cljs.core.async.macros :refer [go]]))

(defn -save [{:keys [instance] :as adapter} query]
  (let [c (async/chan)]
    (.put instance (clj->js query)
          #(let [response (if %1
                            (do
                              (log/log "ERROR: " (.stringify js/JSON %1))
                              [:error query])
                            [:success %2])]
             (when (= :error response)
               (log/log "Error Saving Item: " query))
             (async/put! c response)
             (async/close! c)))
    c))

(defn -update [{:keys [instance] :as adapter} query]
  (let [c (async/chan)]
    (.update instance (clj->js query)
          #(let [response (if %1
                            (do
                              (log/log "ERROR: " (.stringify js/JSON %1))
                              [:error query])
                            [:success %2])]
             (when (= :error response)
               (log/log "Error Updating Item: " query))
             (async/put! c response)
             (async/close! c)))
    c))

(defmulti perform (fn [_ [action-type _]] action-type))

(defmethod perform :update [{:keys [table-names] :as this} [_ payload]]
  (go
    (let [{:keys [course-id revision checkpoint-index tags resource-url]} payload
          payload-type (sp/resolve payload)
          table-name (payload-type table-names)
          delta {:TableName table-name
                 :Key {:course-id course-id
                       :revision revision}
                 :UpdateExpression (str "SET #cp[" checkpoint-index "].#tf = :t, #cp[" checkpoint-index "].#uf = :u")
                 :ExpressionAttributeNames {"#cp" "checkpoints"
                                            "#tf" "tags"
                                            "#uf" "resource-url"}
                 :ExpressionAttributeValues {":t" tags
                                             ":u" resource-url}
                 :ReturnValues "ALL_NEW"}
          res (async/<! (-update this delta))]
      res)))

(defmethod perform :put [{:keys [table-names] :as this} [_ payload]]
  (go
    (let [payload       (payload/create (into [] payload))
          queries       (cv/to-db payload table-names)
          payload-chans (async/merge (map #(-save this %) queries))
          res           (async/<! (async/into [] payload-chans))
          errors        (filter (fn [[result data]] (= :error result)) res)]
      (log/log "Y" (clj->js queries))
      (if (empty? errors)
        {:success true}
        {:error (map second errors)}))))
