(ns backend-adapters.dynamodb.perform
  (:require [backend-adapters.dynamodb.impl :as impl]
            [cljs.core.async :as async]
            [shared.models.error.index :as error]
            [shared.models.payload.index :as payload]
            [shared.protocols.convertible :as cv]
            [shared.protocols.specced :as sp]
            [clojure.walk :as walk]
            [shared.protocols.loggable :as log]
            [cljs.spec.alpha :as spec])
  (:require-macros [cljs.core.async.macros :refer [go]]))

(defn replaceEmptyStrings [obj]
  (walk/postwalk-replace {"" nil} obj))

(defmulti perform (fn [_ [action-type _]] action-type))

(defmethod perform :update [{:keys [table-names] :as this} [_ payload]]
  (let [{:keys [course-id revision checkpoint-index tags resource-url]} payload
        payload-type (sp/resolve payload)
        table-name (payload-type table-names)
        delta {:TableName table-name
               :Key {:course-id course-id
                     :revision revision}
               :UpdateExpression (str "SET #cp[" checkpoint-index "].#tf = :t, #cp[" checkpoint-index "].#uf = :u")
               :ExpressionAttributeNames {"#cp" "checkpoints"
                                          "#tf" "tags"
                                          "#uf" "resource-url"}
               :ExpressionAttributeValues {":t" tags
                                           ":u" resource-url}
               :ReturnValues "ALL_NEW"}]
    (impl/update this delta)))

(defn save-many [{:keys [table-names] :as this} payload]
  (let [payload-type (sp/resolve payload)
        table-name (payload-type table-names)
        items (map (fn [item] {:PutRequest {:Item (-> item replaceEmptyStrings clj->js)}}) payload)
        query {:RequestItems {table-name items}}]
    (impl/batch-put this query)))

(defn save-one [{:keys [table-names] :as this} payload]
  (let [payload-type (sp/resolve payload)
        table-name (payload-type table-names)
        item {:TableName table-name
              :Item (-> payload replaceEmptyStrings clj->js)}]
    (impl/put this item)))

(defmethod perform :save [this [_ payload]]
  (if (spec/valid? (spec/coll-of map?) payload)
    (save-many this payload)
    (save-one this payload)))
