(ns backend-adapters.s3.converters.to-action
  (:require [backend-shared.crypto :as crypto]
            [shared.protocols.specced :as sp]))

(defmulti to-action (fn [_ action] (sp/resolve action)))

(defmethod to-action [:save :raw-user] [{:keys [bucket-names]} [_ user]]
  {:Bucket (:raw-user bucket-names)
   :Key (-> user :profile :user-name)
   :Body (.stringify js/JSON (clj->js user))})

(defmethod to-action [:save :resource] [{:keys [bucket-names]} [_ resource]]
  {:Bucket (:resource bucket-names)
   :Key (str (crypto/hasher (:resource-url resource)) ".json")
   :Body (.stringify js/JSON (clj->js resource))})

(defmethod to-action [:save :portrait] [{:keys [bucket-names]} [_ portrait]]
  {:Bucket (:portrait bucket-names)
   :Key    (str "portraits/" (:user-name portrait) ".jpg")
   :Body   (js/Buffer. (:portrait-data portrait) "base64")})

(defmethod to-action [:save :augmented-bookmark] [{:keys [bucket-names]} [_ bookmark]]
  {:Bucket (:augmented-bookmark bucket-names)
   :Key    (str "bookmarks/" (:timestamp bookmark) "--" (crypto/hasher (:resource-url bookmark)))
   :Body   (.stringify js/JSON (clj->js bookmark))})

(defmethod to-action [:save :raw-resource] [{:keys [bucket-names]} [_ resource]]
  {:Bucket (:raw-resource bucket-names)
   :Key (str (crypto/hasher (:url resource)) ".json")
   :Body (.stringify js/JSON (clj->js resource))})

(defn bookmark->s3 [{:keys [timestamp resource-url] :as bookmark} bucket-name]
  {:Bucket bucket-name
   :Key (str timestamp "--" (crypto/hasher resource-url))
   :Body (.stringify js/JSON (clj->js bookmark))})

(defmethod to-action [:save :bookmarks] [{:keys [bucket-names]} [_ items]]
  (map #(bookmark->s3 % (:bookmarks bucket-names)) items))

(defn repo-item->s3 [{:keys [sha] :as item} bucket-name]
  {:Bucket bucket-name
   :Key (str "repo-items/" sha)
   :Body (.stringify js/JSON (clj->js item))})

(defmethod to-action [:save :repo-items] [{:keys [bucket-names]} [_ items]]
  (map #(repo-item->s3 % (:repo-items bucket-names)) items))
