(ns backend-adapters.dynamodb.converters.query
  (:require [shared.protocols.specced :as sp]
            [clojure.walk :as walk]
            [shared.models.query.index :as query]
            [shared.protocols.loggable :as log]
            [clojure.string :as str]))

(defmulti query->ddb (fn [query _] (sp/resolve query)))

(defmethod query->ddb :course [{:keys [course-id revision] :as query} table-names]
  (let [query-type (sp/resolve query)
        table-name (query-type table-names)]
    (query/create {:TableName table-name
                   :ExpressionAttributeNames {"#ci" "course-id"
                                              "#rev" "revision"}
                   :ExpressionAttributeValues {":civ" course-id
                                               ":revv" (int revision)}
                   :KeyConditionExpression (str "#ci = :civ AND #rev = :revv")})))

(defmethod query->ddb :courses [query table-names]
  (let [query-type (sp/resolve query)
        table-name (query-type table-names)]
    (map #(query/create {:TableName table-name
                         :ExpressionAttributeNames {"#ci" "course-id"
                                                    "#rev" "revision"}
                         :ExpressionAttributeValues {":civ" (str (:course-id %1))
                                                     ":revv" (int (:revision %1))}
                         :KeyConditionExpression (str "#ci = :civ AND #rev = :revv")}) query)))

(defmethod query->ddb :bookmarks [query table-names]
  (let [query-type (sp/resolve query)
        table-name (query-type table-names)]
    (map #(query/create {:TableName table-name
                        :ExpressionAttributeNames {"#fn" "resource-url"}
                         :ExpressionAttributeValues {":val" (str (:resource-url %1))}
                        :KeyConditionExpression (str "#fn = :val")}) query)))

(defmethod query->ddb :resources [query table-names]
  (let [query-type (sp/resolve query)
        table-name (query-type table-names)]
    (map #(query/create {:TableName table-name
                    :Count 1
                    :ExpressionAttributeNames {"#fn" "resource-url"}
                         :ExpressionAttributeValues {":val" (:resource-url %1)}
                    :KeyConditionExpression (str "#fn = :val")}) query)))

(defmethod query->ddb :resource [query table-names]
  (let [query-type (sp/resolve query)
        table-name (query-type table-names)]
    (query/create {:TableName table-name
                   :Count 1
                   :ExpressionAttributeNames {"#fn" "resource-url"}
                   :ExpressionAttributeValues {":val" (:resource-url query)}
                   :KeyConditionExpression (str "#fn = :val")})))

(defmethod query->ddb :profile [query table-names]
  (let [query-type (sp/resolve query)
        table-name (query-type table-names)]
    (query/create {:TableName table-name
                   :Count 1
                   :ExpressionAttributeNames {"#fn" "user-name"}
                   :ExpressionAttributeValues {":val" (str (:user-name query))}
                   :KeyConditionExpression (str "#fn = :val")})))

(defmethod query->ddb :identity [query table-names]
  (let [query-type (sp/resolve query)
        table-name (query-type table-names)]
    (query/create {:TableName table-name
                   :Count 1
                   :ExpressionAttributeNames {"#fn" "auth-id"}
                   :ExpressionAttributeValues {":val" (str (:auth-id query))}
                   :KeyConditionExpression (str "#fn = :val")})))
