(ns backend-adapters.s3.converters.action
  (:require [backend-shared.crypto :as crypto]
            [shared.protocols.specced :as sp]
            [shared.protocols.loggable :as log]))

(defmulti action->s3 (fn [action _] (sp/resolve action)))

(defmethod action->s3 [:save :resource] [[_ resource] bucket-names]
  {:Bucket (:resource bucket-names)
   :Key (str (crypto/hasher (:resource-url resource)) ".json")
   :Body (.stringify js/JSON (clj->js resource))})

(defmethod action->s3 [:save :portrait] [[_ portrait] bucket-names]
  {:Bucket (:portrait bucket-names)
   :Key    (str "portraits/" (:user-name portrait) ".jpg")
   :Body   (js/Buffer. (:portrait-data portrait) "base64")})

(defmethod action->s3 [:save :missing-resource] [[_ bookmark] bucket-names]
  {:Bucket (:missing-resource bucket-names)
   :Key    (str "bookmarks/" (:timestamp bookmark) "--" (crypto/hasher (:resource-url bookmark)))
   :Body   (.stringify js/JSON (clj->js bookmark))})

(defmethod action->s3 [:save :raw-resource] [[_ resource] bucket-names]
  {:Bucket (:raw-resource bucket-names)
   :Key (str (crypto/hasher (:url resource)) ".json")
   :Body (.stringify js/JSON (clj->js resource))})

(defmethod action->s3 [:save :course] [[_ course] bucket-names]
  {:Bucket (:course bucket-names)
   :Key    (str (.now js/Date) ".json")
   :Body   (.stringify js/JSON (clj->js course))})

(defn status->s3 [status bucket-name]
  {:Bucket bucket-name
   :Key    (str (.now js/Date) ".json")
   :Body   (.stringify js/JSON (clj->js status))})

(defmethod action->s3 [:save :status] [[_ status] bucket-names]
  (status->s3 status (:status bucket-names)))

(defmethod action->s3 [:save :statuses] [[_ statuses] bucket-names]
  (map #(status->s3 % (:statuses bucket-names)) statuses))

(defn bookmark->s3 [{:keys [timestamp resource-url] :as bookmark} bucket-name]
  {:Bucket bucket-name
   :Key (str timestamp "--" (crypto/hasher resource-url))
   :Body (.stringify js/JSON (clj->js bookmark))})

(defmethod action->s3 [:save :bookmarks] [[_ bookmarks] bucket-names]
  (map #(bookmark->s3 % (:bookmarks bucket-names)) bookmarks))

(defn repo-item->s3 [{:keys [sha] :as item} bucket-name]
  {:Bucket bucket-name
   :Key (str "repo-items/" sha)
   :Body (.stringify js/JSON (clj->js item))})

(defmethod action->s3 [:save :repo-items] [[_ items] bucket-names]
  (map #(repo-item->s3 % (:repo-items bucket-names)) items))
