(ns shared.models.course.index
  (:require [shared.models.checkpoint.index :as checkpoint]
            [shared.models.course.convertible :as cv-impl]
            [shared.models.course.actionable :as ac-impl]
            [offcourse.specs.course]
            [shared.models.course.get :as get-impl]
            [shared.models.course.missing-data :as md-impl]
            [shared.protocols.actionable :as ac :refer [Actionable]]
            [shared.protocols.convertible :refer [Convertible]]
            [shared.protocols.queryable :refer [Queryable]]
            [shared.protocols.specced :refer [Specced]]
            [shared.protocols.loggable :as log]
            [clojure.set :as set]
            [cljs.spec.alpha :as spec]
            [clojure.string :as c-str]
            [cuerdas.core :as str]))

(defrecord Course []
  Queryable
  (-get [this query] (get-impl/get this query))
  (-missing-data [this query] (md-impl/missing-data this query))
  Actionable
  (-perform [this action] (ac-impl/perform this action))
  Convertible
  (-to-bookmarks [this] (cv-impl/to-bookmarks this))
  (-to-query [this] (cv-impl/to-query this))
  (-to-status-query [this checkpoint] (cv-impl/to-status-query this checkpoint)))

(defn normalize-case [input]
  (-> input c-str/lower-case))

(defn initialize [{:keys [goal curator description checkpoints]}]
  (let [course-id (str (random-uuid))]
    {:base-id course-id
     :course-id course-id
     :repository "offcourse"
     :description description
     :revision 0
     :parent-id nil
     :curator (normalize-case curator)
     :goal (normalize-case goal)
     :timestamp (.now js/Date)
     :checkpoints (map checkpoint/initialize checkpoints)}))

(defmulti create #(if (:courseId %1) :graphql :offcourse))

(defmethod create :graphql [{:keys [baseId courseId goal curator repository description
                                    checkpoints timestamp revision fork parentId]}]
  (-> {:base-id baseId
       :course-id courseId
       :fork-id (:courseId fork)
       :repository repository
       :description description
       :revision revision
       :parent-id parentId
       :curator (normalize-case curator)
       :goal (normalize-case goal)
       :timestamp (js/parseInt timestamp)
       :checkpoints (map checkpoint/create checkpoints)}
      map->Course
      (with-meta {:spec :course/valid})))

(defmethod create :offcourse [{:keys [base-id course-id goal curator repository description
                      checkpoints timestamp revision parent-id]}]
  (-> {:base-id base-id
       :course-id course-id
       :repository repository
       :description description
       :revision revision
       :parent-id parent-id
       :curator (normalize-case curator)
       :goal (normalize-case goal)
       :timestamp timestamp
       :checkpoints (map checkpoint/create checkpoints)}
      map->Course
      (with-meta {:spec :course/valid})))

