(ns ui-components.forms.new-course
  (:require [reagent.core :as r]
            [shared.protocols.actionable :as ac]
            [shared.protocols.specced :as sp]
            [ui-components.forms.sections.checkpoints :refer [CheckpointsSection]]
            [ui-components.forms.sections.controls :refer [FormControls]]
            [ui-components.forms.sections.goal :refer [GoalSection]]
            [ui-components.forms.utils :refer [debounce submit-form]]
            [ui-components.semantic :refer [Form Header Message Segment]]
            [shared.protocols.loggable :as log]
            [shared.protocols.convertible :as cv]
            [shared.models.course.index :as course]
            [shared.models.query.index :as query]))

(defn create-checkpoint [local-state]
  (swap! local-state #(ac/perform %1 [:add :new-checkpoint])))

(defn update-checkpoint [local-state checkpoint]
  (swap! local-state #(ac/perform %1 [:update checkpoint])))

(defn update-goal [local-state new-goal]
  (swap! local-state assoc :goal new-goal))

(defn NewCourseForm [{:keys [course on-cancel on-change on-submit]}]
  (let [local-state (r/atom course)
        update-goal (comp (debounce on-change 1000) update-goal)]
    (fn [{:keys [status]}]
      (let [{:keys [goal checkpoints] :as course} @local-state
            valid?            (sp/valid? course)
            saveable?         (and valid? (= status :checked))
            errors            (sp/errors course)
            query             (query/create course)
            invalid-fields    (into #{} (mapcat :in (-> errors first second)))]
        [Segment {:basic true}
         [Header "Create Course"]
         [Form {:error (= status :rejected)
                :onSubmit #(submit-form %1 course on-submit)}
          [Message {:error true} "You already have a course with this title!"]
          [GoalSection {:goal goal
                        :invalid? (contains? invalid-fields :goal)
                        :check   #(when-not (empty? goal) (on-change query))
                        :update  (partial update-goal local-state)}]
          [CheckpointsSection {:create-checkpoint #(create-checkpoint local-state)
                               :update-checkpoint (partial update-checkpoint local-state)
                               :checkpoints checkpoints}]
          [FormControls {:on-cancel on-cancel
                         :loading? (= status :checking)
                         :disabled? (not saveable?)}]]]))))
