(ns vega-viewer.components.vega-viewer
  (:require [cljsjs.vega]
            [om.core :as om :include-macros true]
            [sablono.core :refer-macros [html]]))

(defn set-resize-handler
  [owner]
  (let [container (om/get-node owner "vega-container")
        resize-handler
        (.addEventListener js/window "resize"
                           (fn [event]
                             (om/set-state! owner
                                            :container-width
                                            (-  (.-offsetWidth container)
                                                30))))]
    (om/set-state! owner :resize-handler resize-handler)))

(defn vega-viewer
  [vega-spec owner]
  (reify
    om/IWillUnmount
    (will-unmount [_]
      (let [{:keys [resize-handler]} (om/get-state owner)]
        (.removeEventListener js/window "resize" resize-handler)))

    om/IDidUpdate
    (did-update [_ _ _]
      (let [vega-container (om/get-node owner "vega-container")
            {:keys [container-width]} (om/get-state owner)
            vega-spec-as-js (clj->js (if-let [{:keys [width]} vega-spec]
                                       vega-spec
                                       (assoc vega-spec
                                              :width
                                              (- container-width
                                                 30))))]
        (js/vg.parse.spec vega-spec-as-js
                          (fn [chart]
                            (let [view (chart #js {:el vega-container})]
                              (.update view))))))

    om/IDidMount
    (did-mount [_]
      (set-resize-handler owner)
      (let [vega-container (om/get-node owner "vega-container")
            container-width (.-offsetWidth vega-container)
            vega-spec-as-js (clj->js (if-let [{:keys [width]} vega-spec]
                                       vega-spec
                                       (assoc vega-spec
                                              :width
                                              (- container-width
                                                 30))))]
        (js/vg.parse.spec vega-spec-as-js
                          (fn [chart]
                            (let [view (chart #js {:el vega-container})]
                              (.update view))))))
    om/IRender
    (render [_]
      (html
       [:div {:ref "vega-container"
              :style {:width "100%"}}]))))
