(ns bundle-reader.core
  (:require [clojure.edn :as edn]
            [clojure.java.io :as jio]))

(defrecord Component [name resources])
(defrecord ComponentReference [name])
(defrecord BundleReference [name])
(defrecord Resources [paths])

(defn read-component [input]
  (cond (vector? input) (let [[comp-name & others] input]
                          [comp-name (->Component comp-name (into [] others))])
        (string? input) (->ComponentReference input)
        :else (throw (ex-info "Unknown data type for component." {:got input}))))

(defn read-bundles [input]
  (cond (map? input) input
        (string? input) (->BundleReference input)
        :else (throw (ex-info "Unknown data type for bundle." {:got input}))))

(defn merged-map [f coll]
  (let [merge-entry (fn [m [k v]]
                      (if (contains? m k)
                        (assoc m k (f (get m k) v))
                        (assoc m k v)))]
    (reduce merge-entry {} coll)))

;; TODO: implement memoization on second argument (by returning a memoized fn)
;; TODO: make it more efficient
(defn flatten-component [components target]
  (cond (instance? ComponentReference target)
        (flatten-component components (get components (:name target)))
        (instance? Component target)
        (let [flattened (mapcat (partial flatten-component components) (:resources target))]
          (->> flattened
               (partition 2)
               (merged-map (comp vec concat))
               (mapcat identity)))
        :else [target]))

;; TODO: watch out for relative paths while removing duplicates... (when you actually implement it)
(defn flatten-bundles [bundles target]
  (cond (instance? BundleReference target)
        (distinct (mapcat (partial flatten-bundles bundles)
                          (get bundles (:name target))))
        ;; TODO: add support for regexes.
        (or (string? target) (instance? java.util.regex.Pattern target)) [target]))

(defn read-all [edn-str]
  (let [res (edn/read-string {:readers {'bundle read-bundles, 'component read-component,
                                        'x      re-pattern, 'resources ->Resources}}
                             edn-str)]
    (loop [bundles    {}
           components {}
           resources  []
           remaining  res]
      (if (seq remaining)
        (let [c    (first remaining)
              nrem (rest remaining)]
          (cond (instance? Resources c) (recur bundles components (apply conj resources (:paths c)) nrem)
                (map? c) (recur (merge bundles c) components resources nrem)
                (vector? c) (recur bundles (conj components c) resources nrem)))
        [bundles components resources]))))

(defn expand-bundles-in-components [bundles flattened-component]
  (into {} (map vec (partition 2 (map (fn [v] (if (vector? v)
                                                (vec (mapcat #(cond (instance? BundleReference %)
                                                                    (flatten-bundles bundles %)
                                                                    :else [%]) v))
                                                v)) flattened-component)))))

(defn map-values [f the-map]
  (into {} (mapv (fn [[k v]] (vector k (f v))) the-map)))

(defn bake-components [bundles components needed-components]
  (->> (for [c needed-components] [c (->ComponentReference c)])
       (map-values (partial flatten-component components))
       (map-values (partial expand-bundles-in-components bundles))))

(defn bake-bundles [bundles needed-bundles]
  (map-values (comp vec (partial flatten-bundles bundles))
              (for [b needed-bundles] [b (->BundleReference b)])))

(defn load-from [file needed-bundles needed-components]
  (let [[bundles components resources] (read-all (slurp file))]
    [(bake-bundles bundles needed-bundles)
     (bake-components bundles components needed-components)
     resources]))

(defn component-handle [component-name assets-type]
  (str component-name "-bundled." (name assets-type)))

(defn component-assets [component-name flattened-component]
  (into {} (map #(vector (component-handle component-name
                                           (first %))
                         (second %))
                flattened-component)))

(defn component-assets-by-name [components component-name]
  (component-assets component-name (get components component-name)))

(def ^:private bundles-file-name "bundles.edn")

(defn standard-path [type path]
  (if (= type :file)
    (jio/file path bundles-file-name)
    (java.net.URL. (str (.toString (jio/resource path)) "/" bundles-file-name))))

(defn- strip-trailing-slash [^String path]
  (if (.endsWith path "/")
    (let [ln (.length path)]
      (.substring path 0 (- ln 1)))
    path))

(defn- get-assets- [file path needed-bundles needed-components]
  (let [stripped-path (strip-trailing-slash path)
        [bundles components resources] (load-from file needed-bundles needed-components)]
    [[stripped-path (merge bundles (apply merge (map #(component-assets (first %)
                                                               (second %)) components)))]
     [stripped-path resources]]))

(defn get-assets [path needed-bundles needed-components]
  (get-assets- (standard-path :file path) path needed-bundles needed-components))

(defn get-assets-from-resources [path needed-bundles needed-components]
  (get-assets- (standard-path :res path) path needed-bundles needed-components))