(ns ^:no-doc unReasoner.preprocessing
 (:require [clojure.java.io :as io][clojure.string :as s]
           [unReasoner.model :as mdl]
           [ontology.normalize :as norm][ontology.core :as ont][ontology.axioms :as ax]
           [util.core :as msc]))

(defn- addThingToYesSet
 [set thing]
 (if (not set)
  [#{thing} #{}]
  (let [[y n] set] [(conj y thing) n])))

(defn- dropThingFromYesSet 
 [[y n] thing]
 [(disj y thing) n])

(defn- addThingToNoSet 
 [set thing]
 (if (not set)
  [#{} #{thing}]
  (let [[y n] set] [y (conj n thing)])))

(defn- dropThingFromNoSet 
 [[y n] thing]
 [y (disj n thing)])

(defn- assocIntoWithVal
 ([map val key](assoc map key val))
 ([map val key1 key2](assoc map key1 (assoc (key1 map) key2 val))))

(defn- assocIntoWithFun
 ([map fun key](assoc map key (fun (key map))))
 ([map fun key1 key2](assoc map key1 (assoc (key1 map) key2 (fun (key2 (key1 map)))))))

(defn- assocValIntoYesAndNo
 [val map key1 key2]
 (assocIntoWithFun (assocIntoWithFun map (fn [set] (addThingToYesSet set val)) key1 key2) (fn [set] (addThingToNoSet set (ont/negate val))) key1 (ont/negate key2)))

(defn- selfCyclic
 [antecedentNames consequentNames]
 (loop [antecedentNames antecedentNames]
  (cond 
   (empty? antecedentNames)
    false
   (contains? consequentNames (first antecedentNames))
    true
   :else
    (recur (rest antecedentNames)))))

(defn- willBeCyclicIfDefinitionAdded 
 [model consequentName]
 (let [notConsequent (ont/negate consequentName)]
 (loop [definitions (keys (:classAxiomsUnfolded model))]
  (cond
   (empty? definitions)
    false
   (= (first definitions) consequentName)
    true
   :else
    (recur (rest definitions))
   ))))

(defn removeCycleFromUnfolded
 [model axiom]
 (let [detectedCyclicAxiom (get (:classAxiomsUnfolded model) (:consequent axiom))
       model (reduce #(assocIntoWithFun %1 (fn [set] (addThingToYesSet set %2)) :classAxiomsGeneral (:consequent axiom)) model (get detectedCyclicAxiom 0))
       model (reduce #(assocIntoWithFun %1 (fn [set] (addThingToNoSet set %2)) :classAxiomsGeneral (:consequent axiom)) model (get detectedCyclicAxiom 1))
       model (assoc model :classAxiomsUnfolded (dissoc (:classAxiomsUnfolded model) (:consequent axiom)))]
       ; TODO resolve remaining cycle after removal of obviious problem
  model
        ))

(defn- unfold
 [model class]
 (if (empty? (:classAxiomsUnfolded model))
  class
 (loop [class class
        classes (ont/getClassNames class)]
        (prn "unfoldStart" class (first classes))
  (cond 
   (empty? classes)
    class
   (get (:classAxiomsUnfolded model) (first classes))
    (do (prn "unfolding" model)
    (recur (update class (first classes) #(partial ont/and ((first classes) (:classAxiomsUnfolded model))))(rest classes))
    )
   :else
    (recur class (rest classes))))))

(defn- unfoldPrimitiveDefinitionEquivalence
 [model axiom](prn "primitiveEquivalenceTest" axiom)
 (cond 
  (= :className (:innerType (first (:classes axiom))))
   (assocValIntoYesAndNo (unfold model (norm/getCSNF (first (rest (:classes axiom))))) model :classAxiomsUnfolded (first (:classes axiom)))
  (= :className (:innerType (first (rest (:classes axiom)))))
   (assocValIntoYesAndNo (unfold model (norm/getCSNF (first (:classes axiom)))) model :classAxiomsUnfolded (first (rest (:classes axiom))))
  (= :not (:innerType (first (:classes axiom))))
   (if (= (:innerType (:class (first (:classes axiom)))) :className)
    (assocValIntoYesAndNo (unfold model (norm/getCSNF (first (rest (:classes axiom))))) model :classAxiomsUnfolded (first (:classes axiom))))             
  (= :not (:innerType (first (rest (:classes axiom)))))
   (if (= (:innerType (:class (first (:classes axiom)))) :className)
    (assocValIntoYesAndNo (unfold model (norm/getCSNF (first (:classes axiom)))) model :classAxiomsUnfolded (first (rest (:classes axiom)))))
  :else 
   nil))

(defn- unfoldPrimitiveDefinitionImplication
 [model axiom](prn "primitiveImplicationTest" axiom)
 (case (:innerType (:antecedent axiom))
  :className (if (willBeCyclicIfDefinitionAdded model (:consequent axiom))
              (removeCycleFromUnfolded (assocIntoWithFun model (fn [set] (addThingToYesSet set (unfold model (norm/getCSNF (:consequent axiom))))) :classAxiomsGeneral (norm/getCSNF (:antecedent axiom))) axiom)
              (assocIntoWithFun model (fn [set] (addThingToYesSet set (unfold model (norm/getCSNF (:consequent axiom))))) :classAxiomsUnfolded (norm/getCSNF (:antecedent axiom))))
  :not (if (= (:innerType (:class (:antecedent axiom))) :className) (update-in model [:classAxiomsUnfolded (:class (:antecedent axiom))] addThingToNoSet (unfold model (norm/getCSNF (:consequent axiom)))) nil)
  :or (loop [classes (:classes (:antecedent axiom))
             model model]
       (if (empty? classes)
        model
        (if-let [model (unfoldPrimitiveDefinitionImplication model (assoc axiom :antecedent (first classes)))] (recur (rest classes) model))))
  nil))

(defn- tryToAbsorb
 "(i) Initialise a set G = {¬D, C}, representing the axiom in the form T ⊑ ¬⊓{¬D, C} (i.e. T ⊑ D ⊔ ¬C).

  (ii) If for some A ∈ G there is a primitive definition axiom (A ⊑ C) ∈ T u , then absorb the general axiom into the primitive definition axiom so that it becomes
        
        A ⊑ ⊓{C, ¬⊓(G \\ {A})},

  and exit.

  (iii) If for some A ∈ G there is an axiom (A ≡ D) ∈ Tu , then substitute A ∈ G with D

       G → {D} ∪ G \\ {A},

  and return to step (ii).

  (iv) If for some ¬A ∈ G there is an axiom (A ≡ D) ∈ Tu , then substitute ¬A ∈ G with ¬D

       G → {¬D} ∪ G \\ {¬A},

  and return to step (ii).

  (v) If there is some C ∈ G such that C is of the form ⊓S, then use associativity to simplify G
      
      G → S ∪ G \\ {⊓S},

  and return to step (ii).

  (vi) If there is some C ∈ G such that C is of the form ¬uS, then for every D ∈ S try to absorb (recursively)
     
     {¬D} ∪ G \\ {¬⊓S},

  and exit.

  (vii) Otherwise, the axiom could not be absorbed, so add ¬⊓G to Tg'

      Tg' → Tg' ∪ ¬⊓G,

  and exit."
 [model axiom](prn "absorbStart" axiom)
 (loop [csnfAxiom (norm/getCSNF axiom)
        classSet (disj (ont/getClasses csnfAxiom) csnfAxiom (:class csnfAxiom))
        Tg (:classAxiomsGeneral model)
        Tu (:classAxiomsUnfolded model)]
  nil)
 model)

(defn- tryToUnfoldOrAbsorbImplicationAxiom
 [model axiom]
 (if (selfCyclic (ont/getNames (:antecedent axiom)) (ont/getNames (:consequent axiom)))
  (assocIntoWithFun model (fn [set] (addThingToYesSet set (unfold model (norm/getCSNF (:consequent axiom))))) :classAxiomsGeneral (norm/getCSNF (:antecedent axiom)))
  (if-let [unfoldedInModel (unfoldPrimitiveDefinitionImplication model axiom)] 
   unfoldedInModel
   (tryToAbsorb model axiom))))

(defn- tryToUnfoldOrAbsorbEquivalenceAxiom
 [model axiom]
 (if (selfCyclic (ont/getNames (first (:classes axiom))) (ont/getNames (first (rest (:classes axiom)))))
  (assocValIntoYesAndNo (unfold model (norm/getCSNF (first (rest (:classes axiom))))) model :classAxiomsGeneral (norm/getCSNF (first (:classes axiom))))
  (if-let [unfoldedInModel (unfoldPrimitiveDefinitionImplication model axiom)] 
   unfoldedInModel 
   (tryToAbsorb model axiom))))

(defn process
 ([ontology]
  (reduce process [(update ontology :axioms #{}) mdl/init] (ont/getAxiomsNoAnnotations ontology)))
 ([[ontology model] axiom](prn model)
  (case (:innerType axiom)
   
   ;regular class axiom
   :classImplication [(ont/addAxiom ontology axiom) (tryToUnfoldOrAbsorbImplicationAxiom (mdl/updateModelForAxiom model axiom) axiom)]

   ;syntactically equivalent to regular class axiom
   :roleDomain (let [norm (norm/toSyntacticEquivalent axiom)][(ont/addAxiom ontology axiom) (tryToUnfoldOrAbsorbImplicationAxiom (mdl/updateModelForAxiom model norm) norm)])
   :roleRange (let [norm (norm/toSyntacticEquivalent axiom)][(ont/addAxiom ontology axiom) (tryToUnfoldOrAbsorbImplicationAxiom (mdl/updateModelForAxiom model norm) norm)])
   :functionalRole (let [norm (norm/toSyntacticEquivalent axiom)][(ont/addAxiom ontology axiom) (tryToUnfoldOrAbsorbImplicationAxiom (mdl/updateModelForAxiom model norm) norm)])
   :functionalInverseRole (let [norm (norm/toSyntacticEquivalent axiom)][(ont/addAxiom ontology axiom) (tryToUnfoldOrAbsorbImplicationAxiom (mdl/updateModelForAxiom model norm) norm)])
   :reflexiveRole (let [norm (norm/toSyntacticEquivalent axiom)][(ont/addAxiom ontology axiom) (tryToUnfoldOrAbsorbImplicationAxiom (mdl/updateModelForAxiom model norm) norm)])
   :irreflexiveRole (let [norm (norm/toSyntacticEquivalent axiom)][(ont/addAxiom ontology axiom) (tryToUnfoldOrAbsorbImplicationAxiom (mdl/updateModelForAxiom model norm) norm)])
   :dataRoleDomain (let [norm (norm/toSyntacticEquivalent axiom)][(ont/addAxiom ontology axiom) (tryToUnfoldOrAbsorbImplicationAxiom (mdl/updateModelForAxiom model norm) norm)])
   :dataRoleRange (let [norm (norm/toSyntacticEquivalent axiom)][(ont/addAxiom ontology axiom) (tryToUnfoldOrAbsorbImplicationAxiom (mdl/updateModelForAxiom model norm) norm)])
   :functionalDataRole (let [norm (norm/toSyntacticEquivalent axiom)][(ont/addAxiom ontology axiom) (tryToUnfoldOrAbsorbImplicationAxiom (mdl/updateModelForAxiom model norm) norm)])

   ;syntactically equivalent to sets of class axioms
   :=classes (let [classes (into [] (:classes axiom))
                   classes (cons (cons (peek classes) (cons (first classes) nil)) (if (> (count classes) 2) (partition 2 1 classes) nil))]
              (loop [parts (for [pair classes](ax/=classes pair))
                     [ontology model] [ontology model]]
              (if (empty? parts) 
               [(ont/addAxiom ontology axiom) model]
               (recur (rest parts) [ontology (tryToUnfoldOrAbsorbEquivalenceAxiom (mdl/updateModelForAxiom model (first parts)) (first parts))]))))
   :disjClasses (loop [parts (ont/toSyntacticEquivalent axiom) 
                       [ontology model] [ontology model]]
                (if (empty? parts) 
                 [ontology model] 
                 (recur (rest parts) (process [ontology model] (first parts)))))
   :disjOr (loop [parts (ont/toSyntacticEquivalent axiom) 
                  [ontology model] [ontology model]]
           (if (empty? parts) 
            [ontology model]
            (recur (rest parts) (process [ontology model] (first parts)))))

   ;TODO role axioms
   :roleImplication (throw (Exception. (str  {:type ::undefinedBehaviorForAxiom :axiom axiom})))
   :=roles (throw (Exception. (str  {:type ::undefinedBehaviorForAxiom :axiom axiom}))) ;has syneq set
   :disjRoles (throw (Exception. (str  {:type ::undefinedBehaviorForAxiom :axiom axiom})))
   :inverseRoles (throw (Exception. (str  {:type ::undefinedBehaviorForAxiom :axiom axiom}))) ;has syneq set
   :symmetricRole (throw (Exception. (str  {:type ::undefinedBehaviorForAxiom :axiom axiom}))) ;has syneq
   :transitiveRole (throw (Exception. (str  {:type ::undefinedBehaviorForAxiom :axiom axiom}))) ;has syneq
   :asymmetricRole (throw (Exception. (str  {:type ::undefinedBehaviorForAxiom :axiom axiom})))
   
   ;TODO data role axioms
   :dataRoleImplication (throw (Exception. (str  {:type ::undefinedBehaviorForAxiom :axiom axiom})))
   :=dataRoles (throw (Exception. (str  {:type ::undefinedBehaviorForAxiom :axiom axiom}))) ;has syneq set
   :disjDataRoles (throw (Exception. (str  {:type ::undefinedBehaviorForAxiom :axiom axiom})))
   
   ;facts
   :classFact (let [norm (norm/getCSNF (:class axiom))]
               (if (= (:innerType norm) :and)
                (loop [classes (:classes norm)
                       om [ontology model]]
                 (if (empty? classes)
                  om
                  (recur (rest classes) (process om (assoc axiom :class (first classes))))))
                (if (= (:innerType norm) :not)
                 [(ont/addAxiom ontology axiom) (update-in model [:facts norm] addThingToNoSet (:individual axiom))]
                 [(ont/addAxiom ontology axiom) (update-in model [:facts norm] addThingToYesSet (:individual axiom))])))
   :roleFact  [(ont/addAxiom ontology axiom) (update-in model [:facts (:role axiom)] addThingToYesSet [(:fromIndividual axiom) (:toIndividual axiom)])]
   :notRoleFact  [(ont/addAxiom ontology axiom) (update-in model [:facts (:role axiom)] addThingToNoSet [(:fromIndividual axiom) (:toIndividual axiom)])]
   :dataRoleFact  [(ont/addAxiom ontology axiom) (update-in model [:facts (:dataRole axiom)] addThingToYesSet [(:fromIndividual axiom) (:toLiteral axiom)])]
   :notDataRoleFact  [(ont/addAxiom ontology axiom) (update-in model [:facts (:dataRole axiom)] addThingToNoSet [(:fromIndividual axiom) (:toLiteral axiom)])]
   :=individuals [(ont/addAxiom ontology axiom) (update-in model [:facts :=] addThingToYesSet (:individuals axiom))]
   :!=individuals [(ont/addAxiom ontology axiom) (update-in model [:facts :=] addThingToNoSet (:individuals axiom))]

   ;TODO weird guys
   :declaration (throw (Exception. (str  {:type ::undefinedBehaviorForAxiom :axiom axiom})))
   :newDataType (throw (Exception. (str  {:type ::undefinedBehaviorForAxiom :axiom axiom})))
   :hasKey (throw (Exception. (str  {:type ::undefinedBehaviorForAxiom :axiom axiom})))
   :annotationAxiom (throw (Exception. (str  {:type ::undefinedBehaviorForAxiom :axiom axiom})))
   :dgRule (throw (Exception. (str  {:type ::undefinedBehaviorForAxiom :axiom axiom})))
   :dlSafeRule (throw (Exception. (str  {:type ::undefinedBehaviorForAxiom :axiom axiom})))
   :dgAxiom (throw (Exception. (str  {:type ::undefinedBehaviorForAxiom :axiom axiom}))))))
