(ns org.bituf.clj-argutil
  (:import (javax.naming Binding Context InitialContext))
  (:import (javax.naming NameClassPair NamingEnumeration))
  (:import (clojure.lang Reflector))
  (:use org.bituf.clj-argutil.internal))


;; ===== illegal argument =====

(defn illegal-arg
  "Throw IllegalArgumentException with specified arguments. Use this when you
   encounter bad/invalid parameters."
  [reason & more]
  (throw (IllegalArgumentException. (apply str reason more))))


;; ===== Type conversion =====

(defn as-string
  "Convert to string"
  [x]
  (if (keyword? x) (name x)
    (str x)))


(defn as-verbose-string
  "Convert to verbose string - useful for diagnostics and error messages"
  [x]
  (if-let [y x] (as-string y)
    "<nil>"))


(defn as-vector
  "Convert/wrap as a vector"
  [anything]
  (if (vector? anything) anything
    (if (or (seq? anything) (set? anything)) (into [] anything)
      (if (map? anything) (into [] (vals anything))
        [anything]))))


(defn as-set
  "Convert/wrap as a set"
  [anything]
  (if (set? anything) anything
    (if (or (seq? anything) (vector? anything)) (into #{} anything)
      (if (map? anything) (into #{} (vals anything))
        #{anything}))))


(defn as-map
  "Convert given collection as a map"
  [coll]
  (if (map? coll) coll
    (if (and (coll? coll) (even? (count coll))) (apply array-map coll)
      (illegal-arg (str "Expected collection with size a multiple of 2, but found "
                     (type coll) ": " (as-verbose-string coll))))))


;; ===== "not" prefixed functions =====

(defn not-nil?   "Same as (not (nil? x))"   [x] (not (nil? x)))
(defn not-empty? "Same as (not (empty? x))" [x] (not (empty? x)))


;; ===== Argument/condition assertion =====

(defmacro when-assert-cond
  "Execute body if *assert-cond* flag is true. Function implemention may
  use this macro to conditionally assert args or return value or any condition
  within the code. Function consumers can instead use either of:
  with-assert-cond
  assert-cond-true
  assert-cond-false"
  [& body]
  `(when *assert-cond*
    ~@body))


(defmacro with-assert-cond
  "Execute body of code in a given assert-cond context (true/false). You may
  like to use 'false' in production mode:
    (with-assert-cond false
      ..)"
  [bool & body]
  `(binding [*assert-cond* ~bool]
    ~@body))


(defmacro assert-cond-false
  "Short for
    (with-assert-cond false
      ..)
  See also: with-assert-cond"
  [& body]
  `(with-assert-cond false
    ~@body))


(defmacro assert-cond-true
  "Short for
    (with-assert-cond true
      ..)
  See also: with-assert-cond"
  [& body]
  `(with-assert-cond true
    ~@body))


;; ===== Assertion helpers =====

(defn verify
  "Apply f? (must return Boolean) to arg - return true when asserted true,
  throw exception otherwise."
  [f? arg]
  (assert (fn? f?))
  (if (f? arg) true
    (illegal-arg
      "Invalid argument " (as-verbose-string arg)
      " (Expected: " (:name (meta f?))
      ", Found: " (as-verbose-string (type arg)) ")")))


(defn assert-type
  [item expected-type]
  (assert (not-nil? item))
  (assert (instance? Class expected-type))
  (try
    (assert (isa? (type item) expected-type))
    (catch AssertionError e
      (throw (AssertionError. (str "Expected " expected-type " but found "
                                (type item)))))))


;; ===== Keyword to/from string conversion =====

(defn k-to-colname
  "Convert a keyword to database column name (string) and replace dash with
  underscore."
  [k]
  (let [n (name k)
        s (.replace n "-" "_")]
    s))


(defn colname-to-k
  "Convert database column name (string) to keyword after replacing underscore
  with dash."
  [s]
  (let [n (.replace s "_" "-")]
    (keyword n)))


(defn k-to-camelstr
  "Convert keyword to camel-case string and treat dash as case-changer.
  :-            --> \"\"
  :to-do        --> \"toDo\"       ; dash triggers upper-case
  :to_do        --> \"to_do\"      ; underscore stays intact
  :to-do-       --> \"toDo\"       ; trailing dash is ignored
  :-from-here   --> \"fromHere\"   ; leading dash is ignored too
  :hello--there --> \"helloThere\" ; consecutive dashes are treated as one"
  [k]
  (let [s (name k)
        tokens (filter not-empty? (into [] (.split s "-")))
        ;; ucase1 converts first character to upper case
        ucase1 #(str (Character/toUpperCase (first %))
                  (apply str (rest %)))
        lcase  #(if (not-nil? %) (.toLowerCase %))
        cctoks (map ucase1 tokens)]
    (apply str (lcase (first cctoks)) (rest cctoks))))


(defn camelstr-to-k
  "Given a camel-case string, convert it into a dash-delimited keyword. Upper
  case character triggers insertion of the dash."
  [cs]
  (let [b (StringBuilder.)
        f #(do
             (if (and (Character/isUpperCase %) (not (empty? b)))
               (.append b \-))
             (.append b (Character/toLowerCase %)))
        _ (doall (map f cs))
        a (filter not-empty? (into [] (.split (.toString b) "-")))
        s (apply str (interpose \- a))]
    (keyword s)))


(defn k-to-methodname
  "Given a keyword and a bunch of method-name prefixes (collection of string),
  construct the method name (string). When called with only a keyword as an
  argument, k-to-methodname behaves like k-to-camelstr.
  See also: k-to-camelstr, camelstr-to-k"
  ([k prefixes]
    (let [s (name (camelstr-to-k (name k)))
          n (if (some #(.startsWith s (str % \-)) prefixes) s
              (str (first prefixes) \- s))]
      (k-to-camelstr (keyword n))))
  ([k]
    (k-to-methodname k [""])))


(defn k-to-setter [k] (k-to-methodname k ["set"]))
(defn k-to-getter [k] (k-to-methodname k ["get" "is"]))


;; ===== Reflection (not recommended for performance-critical code) =====


(defn call-specs
  "Accept a common target object, one or more method specs and turn them into
  call specs. A call spec looks like: [target-object method-spec] and a method
  spec looks like: [method-key & args]
  Example:
    (call-specs \"Hello\" [:char-at 0] [:substring 3 4])
    ;; returns
    [[\"Hello\" :char-at 0]
     [\"Hello\" :substring 3 4]]
    ;; another example - no need to wrap no-arg methods into a vector
    (call-specs \"Hello\" [:char-at 0] :to-string)
    ;; returns
    [[\"Hello\" :char-at 0]
     [\"Hello\" :to-string]]"
  [target method-spec & more-method-specs]
  (let [method-specs (into [method-spec] more-method-specs)]
    (into [] (map #(into [] (cons target (as-vector %))) method-specs))))


(defn method
  "Call instance method on the target object. Wrapper for
  Reflector/invokeInstanceMethod (see link):
  http://github.com/richhickey/clojure/blob/master/src/jvm/clojure/lang/Reflector.java
  Short link: http://j.mp/a2Kd9R
  Examples:
    (call-method \"Hello\" :char-at 0)     ; returns \\H
    (call-method \"Hello\" :substring 3 4) ; returns \"l\"
    ;; the call below returns [\\H \"l\"]
    (call-method [[\"Hello\" :char-at 0]
                  [\"Hello\" :substring 3 4]])
    ;; same call as above expressed using target-method-specs
    (call-method (target-method-specs \"Hello\"
                   [:char-at 0]
                   [:substring 3 4]))"
  ([target method-name & args]
    (Reflector/invokeInstanceMethod
      ;; Object target, String methodName, Object[] args
      target
      (if (keyword? method-name) (k-to-methodname method-name)
        (as-string method-name))
      (into-array Object args)))
  ([call-specs]
    (into [] (map #(apply method %) call-specs))))


(defn pojo-fn
  "Wrap a Plain Old Java Object (POJO) into a function that accepts a
  method spec and invokes the method upon execution."
  ([pojo]
    (fn [method-spec]
      (let [[method-name & args] (as-vector method-spec)]
        (apply method pojo method-name args))))
  ([pojo method-name & args]
    (fn [& more-args]
      (apply method pojo method-name (into [] (concat args more-args))))))


(defn setter
  "Call setter method on a target object using args. 'setter' is either a
  keyword or a string.
  Examples:
    (call-setter obj :price 67.88)   ; .setPrice(67.88)   - returns 67.88
    (call-setter obj :min-max 30 75) ; .setMinMax(30, 75) - returns void
    (call-setter obj :auto-commit)   ; .setAutoCommit()   - returns true
    ;; same stuff in a single call - returns [67.88 nil true]
    (call-setter [[obj :price 67.88]
                  [obj :min-max 30 75]
                  [obj :auto-commit]])
    ;; same stuff without repeating the target object - returns [67.88 nil true]
         (call-setter (call-specs obj [[:price 67.88]
                                       [:min-max 30 75]
                                       [:auto-commit]]))"
  ([target setter-name & args]
    (apply method
      target (if (keyword? setter-name) (k-to-setter setter-name)
               setter-name) args))
  ([setter-specs]
    (into [] (map #(apply setter %) setter-specs))))


(defn setter-fn
  "Wrap a Plain Old Java Object (POJO) into a function that accepts a setter
  method spec and invokes the method upon execution."
  ([pojo]
    (fn [method-spec]
      (let [[method-name & args] (as-vector method-spec)]
        (apply method pojo (k-to-setter method-name) args))))
  ([pojo method-name & args]
    (fn [& more-args]
      (apply method pojo (k-to-setter method-name)
        (into [] (concat args more-args))))))


(defn getter
  "Call getter method on a target object. 'getter-name' is either a keyword
  or a string.
  Example:
    (getter obj :price)        ; .getPrice()    - returns 566.89
    (getter obj :item-code)    ; .getItemCode() - returns 634
    (getter obj :is-available) ; .isAvailable() - returns true
    ;; same calls merged into one
    (getter [[obj :price]
             [obj :item-code]
             [obj :is-available]])
    ;; or a shortcut - returns [566.89 634 true]
    (getter (call-specs obj [:price] [:item-code] [:is-available]))
    ;; even shorter
    (getter (call-specs obj :price :item-code :is-available))"
  ([target getter-name & args]
    (apply method
      target (if (keyword? getter-name) (k-to-getter getter-name)
               getter-name) args))
  ([getter-specs]
    (into [] (map #(apply getter %) getter-specs))))


(defn getter-fn
  "Wrap a Plain Old Java Object (POJO) into a function that accepts a getter
  method spec and invokes the method upon execution.
  Example:
    ;; assuming a Person class having getters getName, getAddress and getEmail
    (map (getter-fn person) [:name :address :email])
"
  ([pojo]
    (fn [method-spec]
      (let [[method-name & args] (as-vector method-spec)]
        (apply method pojo (k-to-getter method-name) args))))
  ([pojo method-name & args]
    (fn [& more-args]
      (apply method pojo (k-to-getter method-name)
        (into [] (concat args more-args))))))


;; ===== Properties handling =====

(defn property-map
  "Transform a given Properties instance to a map."
  [^java.util.Properties properties]
  (let [ks (into [] (.stringPropertyNames properties))
        vs (into [] (map #(.getProperty properties %) ks))]
    (zipmap ks vs)))


(defn strkey-to-keyword
  "Given a map with every key a string, convert keys to keywords.
  Input: {\"a\" 10 \"b\" \"20\"}
  Returns: {:a 10 :b \"20\"}"
  [m]
  (assert (map? m))
  (into {} (map #(let [k (first %)]
                   (assert (string? k))
                   [(keyword k) (last %)]) (seq m))))


(defn is-true?
  "Tell whether a given value is equivalent to true."
  [any]
  (if (string? any)
    (let [v (.toLowerCase any)]
      (or
        (= "true" v)
        (= "yes"  v)
        (= "on"   v)))
    (if (number? any)
      (> any 0)
      (true? any))))


;; ===== JNDI functions (tree-printing not recommended for production use) =====

(defmacro with-root-context
  [root-context & body]
  `(do
    (assert (not (nil? ~root-context)))
    (assert (instance? Context ~root-context))
    (binding [*root-context* ~root-context]
      ~@body)))


(defn- increase-indent []
  (swap! *indent* #(+ % 4)))


(defn- decrease-indent []
  (swap! *indent* #(- % 4)))


(defn- print-entry
  [^NameClassPair next-elem]
  (let [indent-str (apply str
                     (take @*indent* (repeat " ")))]
    (if (nil? next-elem) (println indent-str "--> <nil>")
      (println indent-str "-->"
        (.getName next-elem)
        " (" (type next-elem) "->" (.getClassName next-elem) ")"))))


(declare do-print-jndi-tree)


(defn- print-ne
  [^NamingEnumeration ne ^String parent-ctx]
  (loop []
    (when (.hasMoreElements ne)
      (let [next-elem (.nextElement ne)]
        (print-entry next-elem)
        (increase-indent)
        (if (or (instance? Context next-elem)
              (and (instance? NameClassPair next-elem)
                (instance? Context (.getObject next-elem))))
          (do-print-jndi-tree
            (if (zero? (.length parent-ctx))
              (.getName next-elem)
              (str parent-ctx "/" (.getName next-elem))))
          (println "** Not drilling "
            (type (.getObject next-elem))))
        (decrease-indent))
      (recur))))


(defn- do-print-jndi-tree
  [^String ct]
  (assert (not (nil? ct)))
  (if (instance? Context *root-context*)
    (print-ne (.list *root-context* ct) ct)
    (print-entry *root-context*)))


(defn print-jndi-tree
  ([^String ct]
    (binding [*indent* (atom 0)]
      (do-print-jndi-tree ct)))
  ([]
   (print-jndi-tree "")))


(defn jndi-lookup
  ([^Context context k]
    (.lookup context k))
  ([k]
    (jndi-lookup (InitialContext.) k)))


(defn find-jndi-subcontext
  "Find subcontext in a given JNDI context.
  context  JNDI context
  args     string keys"
  [^Context context & args]
  (assert (not (nil? args)))
  (assert (not (some nil? args)))
  (let [lookup-fn (fn [ctx k] (.lookup ctx k))
        new-ctx (reduce lookup-fn context args)]
    new-ctx))
