(ns org.bituf.clj-dbcp
  (:import org.apache.commons.dbcp.BasicDataSource))

(def *show-jdbc-url* false)


(defstruct datasource-args
  :classname :url :username :password)


(defn make-datasource-args
  "Create an instance of 'datasource-args' using given args."
  [classname url username password]
  (struct datasource-args
    classname
    url
    username
    password))


(defn set-max-active!
  "Set max active connections at a time in the pool. You should call this before
  creating any connection from the datasource."
  [^BasicDataSource datasource max-active]
  (doto datasource
    (.setMaxActive max-active))
  datasource)


(defn set-min-max-idle!
  "Set min and max idle connections count in the pool. You should call this
  before creating any connection from the datasource."
  [^BasicDataSource datasource min-idle max-idle]
  (doto datasource
    (.setMinIdle min-idle)
    (.setMaxIdle max-idle))
  datasource)


(defn set-validation-query!
  "Set validation query for the datasource. You must call this *before*
  creating any connection from the datasource."
  [^BasicDataSource datasource ^String validation-query]
  (doto datasource
      (.setValidationQuery validation-query)
      (.setTestOnBorrow  true)
      (.setTestOnReturn  true)
      (.setTestWhileIdle true))
  datasource)


(defn create-datasource
  "Create basic data source (instance of the BasicDataSource class).
  Arguments:
    classname         (String) the fully qualified driver classname
    url               (String) the JDBC URL
    username          (String) the database username
    password          (String) password for the username
    validation-query  (String, optional) to check database connection is valid
  See also:
    http://commons.apache.org/dbcp/apidocs/org/apache/commons/dbcp/BasicDataSource.html"
  ([classname url username password] ; 4 args
    (doto (BasicDataSource.)
      (.setDriverClassName classname)
      (.setUsername username)
      (.setPassword password)
      (.setUrl url)))
  ([datasource-args] ; 1 arg
    (create-datasource
      (:classname datasource-args)
      (:url       datasource-args)
      (:username  datasource-args)
      (:password  datasource-args))))


(defn db-spec
  "Create a db-spec suitable for use by clojure.contrib.sql"
  [datasource]
  {:datasource datasource})


(defn jndi-datasource
  "Lookup JNDI DataSource. Example Tomcat 6 configuration (/WEB-INF/web.xml):
  <resource-ref>
    <description>
      Resource reference to a factory for java.sql.Connection
      instances that may be used for talking to a particular
      database that is configured in the <Context>
      configurartion for the web application.
    </description>
    <res-ref-name>
      jdbc/EmployeeDB
    </res-ref-name>
    <res-type>
      javax.sql.DataSource
    </res-type>
    <res-auth>
      Container
    </res-auth>
  </resource-ref>
  You can fetch this datasource as follows:
    (jndi-datasource \"java:comp/env/jdbc/EmployeeDB\")"
  ([^javax.naming.Context init-ctx ^String resource-ref-name]
    (.lookup init-ctx resource-ref-name))
  ([resource-ref-name]
    (jndi-datasource
      (javax.naming.InitialContext.) resource-ref-name)))


(def ^{:doc "Default name for the database"}
      default-db-name "default")


; === Derby Config ===
;
(defn derby-args
  "Create data source args for the Apache Derby database. Derby can be hosted
  off a file on the file system, or from the classpath (read-only) or a JAR file
  (read-only).
  Arguments:
    db-protocol  (String) directory|memory|classpath|jar (default: directory)
    db-path      (String) for directory: <dir-containing-the-db>
                          for memory:    <db-name>
                          for classpath: /<db-name>
                          for jar:       (<jar-path>)<[dir/]db-name>
  Examples:
    (derby-args \"\"          \"db1\") ; db-protocol=directory assumed
    (derby-args \"directory\" \"london/sales\")
    (derby-args \"memory\"    \"sample\")
    (derby-args \"classpath\" \"/db1\") ; 'db1' is a directory in classpath
    (derby-args \"jar\"       \"(C:/dbs.jar)products/boiledfood\")
    (derby-args nil           \"//localhost/somedb\") ; connects to server
  Maven/Lein/Cake dependencies (or a later version):
    org.apache.derby/derby      /10.6.1.0 - driver for memory and file mode
    org.apache.derby/derbyclient/10.6.1.0 - driver for network mode
    org.apache.derby/derbynet   /10.6.1.0 - for launching the server itself
  See also:
    http://db.apache.org/derby/docs/dev/devguide/cdevdvlp17453.html
    http://db.apache.org/derby/docs/dev/devguide/rdevdvlp22102.html"
  [db-protocol db-path]
  (let [url (str "jdbc:derby:" (if (empty? db-protocol) ""
                                 (str db-protocol ":"))
              db-path)]
    (if *show-jdbc-url*
      (println "\n**** Derby JDBC URL ****\n" url))
    (make-datasource-args
      "org.apache.derby.jdbc.EmbeddedDriver" ; classname: must be in classpath
      url ; url
      "sa" ; username
      "sx" ; password
      )))


(defn derby-datasource
  "Helper function to create a Derby datasource
  See also:
    derby-network-datasource
    derby-filesystem-datasource
    derby-memory-datasource
    derby-args"
  [db-protocol db-path]
  (create-datasource
    (derby-args db-protocol db-path)))


(defn derby-network-datasource
  "Create a network based (connecting to a server) Derby datasource
  See also: derby-datasource, derby-args"
  [host:port db-name]
  (derby-datasource
    nil (format "//%s/%s;create=true;" host:port db-name)))


(defn derby-filesystem-datasource
  "Create a filesystem (directory) based Derby datasource
  See also: derby-datasource, derby-args"
  ([db-path]
    (derby-datasource
      nil (str db-path ";create=true;")))
  ([]
    (derby-filesystem-datasource default-db-name)))


(defn derby-memory-datasource
  "Create an in-memory Derby datasource
  See also: derby-datasource, derby-args"
  ([db-name]
    (derby-datasource
      "memory" (str db-name ";create=true;")))
  ([]
    (derby-memory-datasource
      default-db-name)))


; === H2 config ===
;
(defn h2-args
  "Create datasource args for the H2 database. H2 database can be hosted off
  the memory, a file or a TCP-port.
  Arguments:
    db-protocol  (String) file|mem|tcp
    db-path      (String) for file: <filepath>
                          for mem: [<databaseName>]
                          for tcp: //<server>[:<port>]/[<path>]<databaseName>
                          for tcp: //<host>/<database>
  Example:
    (h2-args \"mem\"  \"\") ; private in-memory database
    (h2-args \"mem\"  \"sample\") ; named in-memory database
    (h2-args \"file\" \"/home/eddie/sample\")
    (h2-args \"tcp\"  \"//localhost:9092/sample\")
    (h2-args \"tcp\"  \"//localhost/sample\")
    (h2-args \"tcp\"  \"//localhost/home/dir/sample\")
  Maven/Lein/Cake Dependencies:
    com.h2database/h2/1.2.140 (or a later version)
  See also:
    http://www.h2database.com/html/features.html"
  [db-protocol db-path]
  (let [url (str "jdbc:h2:" db-protocol ":" db-path)]
    (if *show-jdbc-url*
      (println "\n**** H2 JDBC URL ****\n" url))
    (make-datasource-args
      "org.h2.Driver" ; classname: must be in classpath
      url ; url
      "sa" ; username
      ""   ; password
      )))


(defn h2-datasource
  "Create an H2 data source
  See also: h2-args"
  [db-protocol db-path]
  (create-datasource
    (h2-args db-protocol db-path)))


(defn h2-network-datasource
  "Create a network based (connecting to a server) H2 datasource
  See also: h2-datasource, h2-args"
  [host:port db-path]
  (h2-datasource
    "tcp" (format "//%s/%s" host:port db-path)))


(defn h2-filesystem-datasource
  "Create a filesystem (directory) based H2 datasource
  See also: h2-datasource, h2-args"
  ([db-path]
    (h2-datasource "file" db-path))
  ([]
    (h2-filesystem-datasource default-db-name)))


(defn h2-memory-datasource
  "Create an in-memory H2 datasource
  See also: h2-datasource, h2-args"
  ([db-name]
    (h2-datasource
      "mem" db-name))
  ([]
    (h2-memory-datasource "")))


; === HSQL-DB (HyperSQL) config ===
;
(defn hsql-args
  "Create datasource args for the HyperSQL database. HyperSQL database can be
  hosted off the memory, a file or a TCP-port. Example JDBC URLs it should
  create are these:
    jdbc:hsqldb:hsql://localhost/xdb -- server
    jdbc:hsqldb:file:/opt/db/testdb -- file
    jdbc:hsqldb:mem:aname -- memory
  Arguments:
    db-protocol  (String) file|mem|hsql (hsql = server-mode, default port 9001)
    db-path      (String) for file: <filepath>
                          for mem: [<databaseName>]
                          for tcp: //<server>[:<port>]/[<path>]<databaseName>
                          for tcp: //<host>/<database>
  Example:
    (hsql-args \"mem\"  \"\") ; private in-memory database
    (hsql-args \"mem\"  \"sample\") ; named in-memory database
    (hsql-args \"file\" \"/home/eddie/sample\")
    (hsql-args \"hsql\" \"//localhost:9001/sample\")
    (hsql-args \"hsql\" \"//localhost/sample\")
    (hsql-args \"hsql\" \"//localhost/home/dir/sample\")
  Maven/Lein/Cake Dependencies:
    org.hsqldb/hsqldb/2.0.0 (or a later version)
  See also:
    http://hsqldb.org/doc/guide/ch01.html"
  [db-protocol db-path]
  (let [url (str "jdbc:hsqldb:" db-protocol ":" db-path)]
    (if *show-jdbc-url*
      (println "\n**** HSQL JDBC URL ****\n" url))
    (make-datasource-args
      "org.hsqldb.jdbcDriver" ; classname: must be in classpath
      url ; url
      "sa" ; username
      ""   ; password
      )))


(defn hsql-datasource
  "Create an HSQLDB/HyperSQL data source.
  See also: hsql-args"
  ([db-protocol db-path]
    (create-datasource
      (hsql-args db-protocol db-path)))
  ([]
    (hsql-datasource "mem" default-db-name)))


(defn hsql-network-datasource
  "Create a network based (connecting to a server) HSQLDB/HyperSQL datasource
  See also: hsql-datasource, hsql-args"
  [host:port db-path]
  (hsql-datasource
    "hsql" (format "//%s/%s" host:port db-path)))


(defn hsql-filesystem-datasource
  "Create a filesystem (directory) based HSQLDB/HyperSQL datasource
  See also: hsql-datasource, hsql-args"
  ([db-path]
    (hsql-datasource "file" db-path))
  ([]
    (hsql-filesystem-datasource default-db-name)))


(defn hsql-memory-datasource
  "Create an in-memory HSQLDB/HyperSQL datasource
  See also: hsql-datasource, hsql-args"
  ([db-name]
    (hsql-datasource "mem" db-name))
  ([]
    (hsql-memory-datasource default-db-name)))


; === MySQL config ===
;
(defn mysql-args
  "Create datasource args for the MySQL database.
  Arguments:
    db-host:port  (String) database hostname (optionally followed by :port-no)
    db-name       (String) database name to connect to
    username      (String) database username
    password      (String) password for the database user
  Examples:
    (mysql-args \"localhost\"      \"sales\" \"salesuser\" \"secret\")
    (mysql-args \"localhost:3306\" \"emp\"   \"empuser\"   \"SeCrEt\")"
  [db-host:port db-name username password]
  (make-datasource-args
    "com.mysql.jdbc.Driver" ; classname: must be in classpath
    (str "jdbc:mysql://" db-host:port "/" db-name) ; url
    username ; username
    password ; password
    ))


(defn mysql-datasource
  "Create MySQL data source
  See also: mysql-args"
  [db-host:port db-name username password]
  (let [args (mysql-args
               db-host:port db-name username password)
        ds   (create-datasource args)]
    (set-validation-query! ds
      "SELECT 1;")))


; === PostgreSQL config ===
;
(defn pgsql-args
  "Create datasource args for the PostgreSQL database.
  Arguments:
    db-host:port  (String) database hostname (optionally followed by :port-no)
    db-name       (String) database name to connect to
    username      (String) database username
    password      (String) password for the database user
  Examples:
    (pgsql-args \"localhost\"      \"sales\" \"salesuser\" \"secret\")
    (pgsql-args \"localhost:5432\" \"emp\"   \"empuser\"   \"SeCrEt\")"
  [db-host:port db-name username password]
  (make-datasource-args
    "org.postgresql.Driver" ; classname: must be in classpath
    (str "jdbc:postgresql://" db-host:port "/" db-name) ; url
    username ; username
    password ; password
    ))


(defn pgsql-datasource
  "Create PostgreSQL data source
  See also: pgsql-args"
  [db-host:port db-name username password]
  (let [args (pgsql-args
               db-host:port db-name username password)
        ds   (create-datasource args)]
    (set-validation-query! ds
      "SELECT version();")))


; === Oracle config ===
;
(defn oracle-args
  "Create datasource args for the Oracle database. Oracle JDBC URLs usually
  look like these:
    jdbc:oracle:thin:[user/password]@[host][:port]:SID
    jdbc:oracle:thin:[user/password]@//[host][:port]/SID
  Arguments:
    db-host:port  (String) database hostname (optionally followed by :port-no)
    system-id     (String) system ID to connect to
    username      (String) database username
    password      (String) password for the database user
  Examples:
    (oracle-args \"localhost\"      \"sales\" \"salesuser\" \"secret\")
    (oracle-args \"localhost:1521\" \"emp\"   \"empuser\"   \"SeCrEt\")"
  [db-host:port system-id username password]
  (make-datasource-args
    "oracle.jdbc.driver.OracleDriver" ; classname: must be in classpath
    (str "jdbc:oracle:thin:@//" db-host:port "/" system-id) ; url
    username ; username
    password ; password
    ))


(defn oracle-datasource
  "Create Oracle data source
  See also: oracle-args"
  [db-host:port system-id username password]
  (let [args (oracle-args
               db-host:port system-id username password)
        ds   (create-datasource args)]
    (set-validation-query! ds
      "SELECT 1 FROM DUAL;")))


; === DB2 config ===
;
(defn db2-args
  "Create datasource args for IBM DB2 database (Using Type 4 universal driver).
  DB2 JDBC URLs usually look like these:
    jdbc:db2://<host>[:<port>]/<database_name>
  Arguments:
    db-host:port  (String) database hostname (optionally followed by :port-no)
    db-name       (String) database name to connect to
    username      (String) database username
    password      (String) password for the database user
  Examples:
    (db2-args \"localhost\"       \"sales\" \"salesuser\" \"secret\")
    (db2-args \"localhost:50000\" \"emp\"   \"empuser\"   \"SeCrEt\")"
  [db-host:port db-name username password]
  (make-datasource-args
    "com.ibm.db2.jcc.DB2Driver" ; classname: must be in classpath
    (str "jdbc:db2://" db-host:port "/" db-name) ; url
    username ; username
    password ; password
    ))


(defn db2-datasource
  "Create DB2 data source
  See also: db2-args"
  [db-host:port db-name username password]
  (let [args (db2-args
               db-host:port db-name username password)
        ds   (create-datasource args)]
    (set-validation-query! ds
      "select * from sysibm.SYSDUMMY1;")))


;; ===== jTDS related =====

(defn- props-str
  "Build properties string for jTDS JDBC URL. If the input is a string, return
  as it is. If it is a map, return like this: \";k1=v1;k2=v2...\" "
  [properties]
  (let [as-str #(if (keyword? %) (name %)
                 (str %))]
    (apply str
      (if (map? properties) (map #(format ";%s=%s"
                                    (as-str (first %)) (as-str (last %)))
                              (seq properties))
        properties))))


(defn jtds-args
  "Create datasource args for MS SQL Server database (Using jTDS driver).
  jTDS URLs usually look like these:
    jdbc:jtds:<server-type>://<server>[:<port>][/<database>][;<property>=<value>[;...]]
  Arguments:
    server-type   (String) \"sqlserver\" or \"sybase\"
    db-host:port  (String) database hostname (optionally followed by :port-no)
    db-name       (String, can be nil) database name to connect to
    username      (String) database username
    password      (String) password for the database user
    properties    (String, can be nil) properties delimited by semicolon
                  (map) {k1 v1 k2 v2 ...}
  Examples:
    (jtds-args \"sqlserver\" \"localhost\" \"sales\" \"salesuser\" \"secret\")
    (jtds-args \"sybase\"    \"localhost\" \"emp\"   \"empuser\"   \"SeCrEt\")
  See also: (1) props-str function
            (2) http://jtds.sourceforge.net/faq.html"
  [server-type db-host:port db-name username password properties]
  (make-datasource-args
    "net.sourceforge.jtds.jdbc.Driver" ; classname: must be in classpath
    ;; JDBC URL
    (format "jdbc:jtds:%s://%s%s%s" server-type db-host:port
      (and db-name (str "/" db-name)) ; database name (can be empty)
      (props-str properties)) ; properties (can be empty)
    username ; username
    password ; password
    ))


(defn jtds-datasource
  "Create jTDS data source
  See also: jtds-args"
  [server-type db-host:port db-name username password properties]
  (let [args (jtds-args
               server-type
               db-host:port db-name username password
               properties)
        ds   (create-datasource args)]
    (set-validation-query! ds
      "select 1;")))


; === MS SQL Server config ===
;
(defn sqlserver-datasource
  "Create MS SQL Server data source (default port 1433)
  See also: jtds-args, jtds-datasource
            http://jtds.sourceforge.net/faq.html"
  ([db-host:port db-name username password properties]
    (jtds-datasource
      "sqlserver"
      db-host:port db-name username password properties))
  ([db-host:port db-name username password]
    (sqlserver-datasource
      db-host:port db-name username password nil))
  ([db-host:port username password]
    (sqlserver-datasource
      db-host:port nil username password nil)))


; === Sybase config ===
;
(defn sybase-datasource
  "Create Sybase data source (default port 7100)
  See also: jtds-args, jtds-datasource
            http://jtds.sourceforge.net/faq.html"
  ([db-host:port db-name username password properties]
    (jtds-datasource
      "sybase"
      db-host:port db-name username password properties))
  ([db-host:port db-name username password]
    (sybase-datasource
      db-host:port db-name username password nil))
  ([db-host:port username password]
    (sybase-datasource
      db-host:port nil username password nil)))


; === SQLite-DB config ===
;
(defn sqlite-args
  "Create datasource args for the SQLite database. SQLite database can be
  hosted off the memory, or a file. Example JDBC URLs it should create are:
    jdbc:sqlite:C:/work/mydatabase.db        -- file
    jdbc:sqlite:/home/leo/work/mydatabase.db -- file
    jdbc:sqlite::memory:                     -- memory
  Arguments:
    db-path (String) for file   -- <filepath>
                     for memory -- :memory:
  Example:
    (sqlite-args \"C:/work/mydatabase.db\")
    (sqlite-args \":memory:\")
  Maven/Lein/Cake Dependencies:
    org.xerial/sqlite-jdbc/3.7.2 (or a later version)
  See also:
    http://www.xerial.org/trac/Xerial/wiki/SQLiteJDBC"
  [db-path]
  (let [url (str "jdbc:sqlite:" db-path)]
    (if *show-jdbc-url*
      (println "\n**** SQLite JDBC URL ****\n" url))
    (make-datasource-args
      "org.sqlite.JDBC" ; classname: must be in classpath
      url ; url
      "sa" ; username
      ""   ; password
      )))


(defn sqlite-datasource
  "Create an SQLite data source.
  See also: sqlite-args"
  ([db-path]
    (create-datasource (sqlite-args db-path)))
  ([]
    (sqlite-datasource ":memory:")))


(defn sqlite-filesystem-datasource
  "Create a filesystem (file) based SQLite datasource.
  See also: sqlite-datasource, sqlite-args"
  ([db-path]
    (sqlite-datasource db-path))
  ([]
    (sqlite-datasource (str "sqlite" default-db-name))))


(defn sqlite-memory-datasource
  "Create an in-memory SQLite datasource.
  See also: sqlite-datasource, sqlite-args"
  ([]
    (sqlite-datasource)))


; === CUBRID config ===
;
(defn cubrid-args
  "Create datasource args for the CUBRID database. The default database port is
  8001 but the JDBC connection typically happens through a broker at port 33000
  or 30000. Example JDBC URL:
    jdbc:cubrid:localhost:33000:demodb:::
  Arguments:
    db-host:port  (String) database hostname and :port-no (colon, then port-no)
    db-name       (String) database name to connect to
    username      (String) database username
    password      (String) password for the database user
  Examples:
    (cubrid-args \"localhost:33000\" \"sales\" \"salesuser\" \"secret\")
    (cubrid-args \"localhost:30000\" \"emp\"   \"empuser\"   \"SeCrEt\")
  See also: http://wiki.cubrid.org/index.php/CUBRID_Manuals/cubrid_2008_R3.0_manual/Getting_Started_with_CUBRID/JDBC_Sample
            http://j.mp/fS5Evv (short URL of the above)"
  [db-host:port db-name username password]
  (make-datasource-args
    "cubrid.jdbc.driver.CUBRIDDriver" ; classname: must be in classpath
    (format "jdbc:cubrid:%s:%s:::" db-host:port db-name) ; url
    username ; username
    password ; password
    ))


(defn cubrid-datasource
  "Create CUBRID data source
  See also: cubrid-args"
  [db-host:port db-name username password]
  (let [args (cubrid-args
               db-host:port db-name username password)
        ds   (create-datasource args)]
    (set-validation-query! ds
      "SELECT 1;")))
