(ns boot-javac-star.javac
  (:require [boot.core :as core]
            [boot.pod  :as pod]
            [boot.util :as util]
            [clojure.java.io :as io]
            [clojure.string :as str]
            [clojure.set :as set])
  (:import (java.io ByteArrayOutputStream)
           (java.util.concurrent ConcurrentHashMap)
           (clojure.lang DynamicClassLoader)
           (javax.tools ForwardingJavaFileManager
                        JavaCompiler
                        JavaFileObject$Kind
                        SimpleJavaFileObject
                        ToolProvider)))

(defn construct-class-name [prefix path]
  (let [path-seq (-> path
                     (str/replace prefix "")
                     (str/replace #"\.java$" "")
                     (str/split #"/"))]
    (->> path-seq
         (remove empty?)
         (interpose ".")
         str/join)))

(def ^ConcurrentHashMap class-cache
  (-> (.getDeclaredField clojure.lang.DynamicClassLoader "classCache")
      (doto (.setAccessible true))
      (.get nil)))

(defn source-object
  [class-name source]
  (proxy [SimpleJavaFileObject]
      [(java.net.URI/create (str "string:///"
                                 (.replace ^String class-name \. \/)
                                 (. JavaFileObject$Kind/SOURCE extension)))
       JavaFileObject$Kind/SOURCE]
    (getCharContent [_] source)))

(defn class-object
  "Returns a JavaFileObject to store a class file's bytecode."
  [class-name baos]
  (proxy [SimpleJavaFileObject]
      [(java.net.URI/create (str "string:///"
                                 (.replace ^String class-name \. \/)
                                 (. JavaFileObject$Kind/CLASS extension)))
       JavaFileObject$Kind/CLASS]
    (openOutputStream [] baos)))

(defn class-manager
  [cl manager cache]
  (proxy [ForwardingJavaFileManager] [manager]
    (getClassLoader [location]
      cl)
    (getJavaFileForOutput [location class-name kind sibling]
      (let [baos (ByteArrayOutputStream.)]
        (swap! cache assoc class-name baos)
        (class-object class-name baos)))))

(defn source->bytecode [opts diag-coll class-name source]
  (let [compiler   (or (ToolProvider/getSystemJavaCompiler)
                       (throw (Exception. "The java compiler is not working. Please make sure you use a JDK!")))
        cache    (atom {})
        mgr      (class-manager nil (.getStandardFileManager compiler nil nil nil) cache)
        task     (.getTask compiler nil mgr diag-coll opts nil [(source-object class-name source)])]
    (when (.call task)
      (zipmap
       (keys @cache)
       (->> @cache
            vals
            (map #(.toByteArray ^ByteArrayOutputStream %)))))))

(defn compile-java
  [opts diag-coll class-name source already-defined]
  (let [cl (clojure.lang.RT/makeClassLoader)
        class->bytecode (source->bytecode opts diag-coll class-name source)]
    (doseq [[class-name bytecode] class->bytecode
            :when (not (@already-defined class-name))]
      (println (format "Compiling %s (%s)" class-name cl))
      (.remove class-cache class-name)
      (.defineClass ^DynamicClassLoader cl class-name bytecode nil))
    (swap! already-defined into (keys class->bytecode))
    (keys class->bytecode)))
