(ns pallet.box.clojure-dev
  "Set up a clojure development box"
  (:require
   [pallet.action.exec-script :as exec-script]
   [pallet.action.remote-file :as remote-file]
   [pallet.core :as core]
   [pallet.crate.automated-admin-user :as automated-admin-user]
   [pallet.crate.git :as git]
   [pallet.crate.java :as java]
   [pallet.crate.maven :as maven]))

(defn lein
  "Install latest stable lein"
  [session]
  (->
   session
   (remote-file/remote-file
    "/usr/local/bin/lein"
    :url "https://github.com/technomancy/leiningen/raw/stable/bin/lein"
    :mode "0755")))

(defn maven
  "Install maven"
  [session]
  (-> session (maven/package)))

(defn git
  "Install git"
  [session]
  (-> session (git/download :version "3.0.3")))

(def emacs-clj
  "(add-to-list 'load-path \"~/.elisp\")
(require 'package)
(add-to-list 'package-archives
   '(\"marmalade\" . \"http://marmalade-repo.org/packages/\"))")

(def
  ^{:doc "url for package.el taken from marmalade's home page."}
  package-el-url
  "http://repo.or.cz/w/emacs.git/blob_plain/1a0a666f941c99882093d7bd08ced15033bc3f0c:/lisp/emacs-lisp/package.el")

(defn emacs
  "Install emacs"
  [session]
  (let [user (:username (session/admin-user session))]
    (->
     session
     (package/package "emacs")
     (directory/directory ".elisp" :owner user)
     (remote-file/remote-file
      ".elisp/package.el" :url package-el-url :owner user)
     (remote-file/remote-file
      ".emacs-clj" :content emacs-clj :owner user)
     (exec-script/exec-checked-script
      "Install package.el"
      (if-not (fgrep ".emacs-clj" ".emacs")
        (pipe
         (println "\n(load (expand-file-name \"~/.emacs-clj\"))")
         (cat >> ".emacs"))))
     (exec-script/exec-checked-script
      "Install clojure-mode"
      (emacs -q -batch -eval (quoted "(package-install \"clojure-mode\")"))))))

(when
    (load
     (expand-file-name "~/.emacs.d/elpa/package.el"))
  (add-to-list
   'package-archives
   '("marmalade" . "http://marmalade-repo.org/packages/"))
  (package-initialize))


(defn base
  "Base clojure-dev box"
  [session]
  (java/java)
  (git)
  (maven)
  (lein))

(def clojure-dev-server
  (core/server-spec
   :phases {:bootstrap automated-admin-user
            :configure base}))

(def clojure-dev
  (core/group-spec
   "clojuredev"
   :extends [clojure-dev-server]))
