(ns pallet.crate.package-repository
  "Crate for setting up a package repository.

   The model here is to create a repository with (repository path),
   and then upload content to it with (upload :options-as-for-remote-directory).
   There will also an upload-to, that can be used to upload from a
   different machine to the repository host."
  (:require
   [pallet.resource.package :as package]
   [pallet.resource.directory :as directory]
   [pallet.resource.exec-script :as exec-script]
   [pallet.resource.format :as format]
   [pallet.resource.remote-directory :as remote-directory]
   [pallet.resource.remote-file :as remote-file]
   [pallet.resource.user :as user]
   [pallet.stevedore :as stevedore]
   [pallet.parameter :as parameter]
   [pallet.request-map :as request-map]
   [pallet.script :as script])
  (:use
   pallet.thread-expr))

(def repository-owner "repoman")
(def repository-group "repoman")

(script/defscript new-repository-content [path])
(stevedore/defimpl new-repository-content [#{:aptitude}] [path]
  (dput local "*.changes"))
(stevedore/defimpl new-repository-content [#{:yum}] [path]
  (createrepo ~path))

(defmulti default-upload-path
  "Based on a repository filesystem path, calculate an upload path for
   incoming packages."
  (fn [request & _] (:target-packager request)))

(defmethod default-upload-path :yum
  [request path]
  path)

(defmethod default-upload-path :aptitude
  [request path]
  (str path "/mini-dinstall/incoming"))

(defmulti config-files
  "Create any required configuration files."
  (fn [request & _] (:target-packager request)))

(defmethod config-files :default [request path options] nil)

(defmethod config-files :aptitude
  [request path options]
  (let [path (parameter/get-for-target request [:package-repository :path])
        upload-path (parameter/get-for-target
                     request [:package-repository :upload-path])]
    (->
     request
     (remote-file/remote-file
      "$HOME/.dput.cf"
      :content (format/sectioned-properties
                (merge
                 {:local
                  {:fqdn "localhost"
                   :method "local"
                   :incoming upload-path
                   :allow_unsigned_uploads 1
                   :post_upload_command "mini-dinstall --batch"}}
                 (:dput options))))
     (remote-file/remote-file
      "$HOME/.mini-dinstall.conf"
      :content (format/sectioned-properties
                (merge
                 {:DEFAULT
                  {:architectures  "all, i386"
                   :archivedir path
                   :use_dnotify 0
                   :verify_sigs 0
                   :extra_keyrings "$HOME/.gnupg/pubring.gpg"
                   :mail_on_success 0
                   :archive_style "flat"
                   :poll_time 40
                   :mail_log_level "NONE"
                   :generate_release 1
                   :release_description "packages"}
                  :lucid {}
                  :karmic {}}
                 (:mini-dinstall options)))))))

(defn repository
  "Create a package repository.  The path should be supplied as a local path.
   Note that this creates a local repository. If you wish to serve it, then
   you will need to install a web server a configure it to serve path."
  [request path & {:keys [upload-path owner group mode]
                   :or {owner repository-owner
                        group repository-group
                        mode "0755"}}]
  (let [upload-path (or upload-path (default-upload-path request path))]
    (->
     request
     (package/packages :yum ["createrepo"] :aptitude ["dput" "mini-dinstall"])
     (user/group group :system true)
     (user/user owner :system true :shell "/bin/false")
     (directory/directory path :owner owner :group group :mode mode)
     (when->
      (not= path upload-path)
      (directory/directory upload-path :owner owner :group group :mode mode))
     (parameter/assoc-for-target
      [:package-repository :owner] owner
      [:package-repository :group] group
      [:package-repository :path] path
      [:package-repository :upload-path] upload-path
      [:package-repository :ip] (request-map/target-ip request)))))

(defn upload
  "Upload package to the repository.
   Options are as for remote-directory."
  [request & {:as options}]
  (let [owner (parameter/get-for-target request [:package-repository :owner])
        group (parameter/get-for-target request [:package-repository :group])
        path (parameter/get-for-target request [:package-repository :path])
        upload-path (parameter/get-for-target
                     request [:package-repository :upload-path])]
    (->
     request
     (apply-map->
      remote-directory/remote-directory
      upload-path :owner owner :group group options)
     (exec-script/exec-checked-script
      "Update repository for new content"
      (new-repository-content ~path)))))

;; (defn upload-to
;;   "Upload package from the target machine to the repository host.
;;    This requires setting up an ssh key on the target, and enabling it on
;;    the repository host."
;;   [request & {:as options}])
