(ns pallet.crate.ssh-key
  (:require
   [pallet.stevedore :as stevedore]
   [pallet.utils :as utils]
   [pallet.parameter :as parameter]
   [pallet.resource :as resource]
   [pallet.resource.directory :as directory]
   [pallet.resource.exec-script :as exec-script]
   [pallet.resource.remote-file :as remote-file]
   [pallet.resource.file :as file]
   [clojure.string :as string])
  (:use
   [pallet.target :only [admin-group]]
   [pallet.resource.user :only [user-home]]
   [pallet.resource.file :only [chmod chown]]
   pallet.thread-expr))

(defn user-ssh-dir [user]
  (str (stevedore/script (user-home ~user)) "/.ssh/"))

(defn authorize-key
  "Authorize a public key on the specified user."
  [request user public-key-string & {:keys [authorize-for-user] :as options}]
  (let [target-user (or authorize-for-user user)
        dir (user-ssh-dir target-user)
        auth-file (str dir "authorized_keys")]
    (->
     request
     (directory/directory dir :owner target-user :mode "755")
     (file/file auth-file :owner target-user :mode "644")
     (exec-script/exec-checked-script
      "authorize-key"
      (var auth_file ~auth-file)
      (if-not (fgrep (quoted ~(string/trim public-key-string)) @auth_file)
        (echo (quoted ~public-key-string) ">>" @auth_file))))))

(defn authorize-key-for-localhost
  "Authorize a user's public key on the specified user, for ssh access to
  localhost.  The :authorize-for-user option can be used to specify the
  user to who's authorized_keys file is modified."
  [request user public-key-filename & {:keys [authorize-for-user] :as options}]
  (let [target-user (or authorize-for-user user)
        key-file (str (user-ssh-dir user) public-key-filename)
        auth-file (str (user-ssh-dir target-user) "authorized_keys")]
    (->
     request
     (directory/directory
      (user-ssh-dir target-user) :owner target-user :mode "755")
     (file/file auth-file :owner target-user :mode "644")
     (exec-script/exec-checked-script
      "authorize-key"
      (var key_file ~key-file)
      (var auth_file ~auth-file)
      (if-not (grep (quoted @(cat @key_file)) @auth_file)
        (do
          (echo -n (quoted "from=\\\"localhost\\\" ") ">>" @auth_file)
          (cat @key_file ">>" @auth_file)))))))

(defn install-key
  "Install a ssh private key."
  [request user key-name private-key-string public-key-string]
  (let [ssh-dir (user-ssh-dir user)]
    (->
     request
     (directory/directory ssh-dir :owner user :mode "755")
     (remote-file/remote-file
      (str ssh-dir key-name)
      :owner user :mode "600"
      :content private-key-string)
     (remote-file/remote-file
      (str ssh-dir key-name ".pub")
      :owner user :mode "644"
      :content public-key-string))))

(def ssh-default-filenames
     {"rsa1" "identity"
      "rsa" "id_rsa"
      "dsa" "id_dsa"})

(defn generate-key
  "Generate an ssh key pair for the given user, unless one already
   exists. Options are:
     :file path     -- output file name (within ~user/.ssh directory)
     :type key-type -- key type selection
     :no-dir true   -- do note ensure directory exists
     :passphrase    -- new passphrase for encrypring the private key
     :comment       -- comment for new key"
  [request user & {:keys [type file passphrase no-dir comment]
                   :or {type "rsa" passphrase ""}
                   :as  options}]
  (let [key-type type
        path (stevedore/script
              ~(str (user-ssh-dir user)
                    (or file (ssh-default-filenames key-type))))
        ssh-dir (.getParent (java.io.File. path))]
    (->
     request
     (when-not-> (or (:no-dir options))
                 (directory/directory ssh-dir :owner user :mode "755"))
     (exec-script/exec-checked-script
      "ssh-keygen"
      (var key_path ~path)
      (if-not (file-exists? @key_path)
        (ssh-keygen ~(stevedore/map-to-arg-string
                      {:f (stevedore/script @key_path)
                       :t key-type
                       :N (str \" passphrase \")
                       :C (str
                           \"
                           (or (:comment options "generated by pallet"))
                           \")}))))
     (file/file path :owner user :mode "0600")
     (file/file (str path ".pub") :owner user :mode "0644"))))

(defn record-public-key
  [request user & {:keys [filename type]
                   :or {type "rsa"} :as options}]
  (let [filename (or filename (ssh-default-filenames type))
        path (str (user-ssh-dir user) filename ".pub")]
    (->
     request
     (remote-file/with-remote-file
       (fn [request local-path]
         (resource/as-local-resource
          request
          (fn [request]
            (parameter/assoc-for-target
             request [:user (keyword user) (keyword filename)]
             (slurp local-path)))))
       path))))

#_
(pallet.core/defnode a {}
  :bootstrap (pallet.resource/phase
              (pallet.crate.automated-admin-user/automated-admin-user))
  :configure (pallet.resource/phase
              (pallet.crate.ssh-key/generate-key "duncan")
              (pallet.crate.ssh-key/record-public-key "duncan")))
