(ns doctordoc.core
  "Documentation controller"
  (:use ring.util.response)
  (:require
   [net.cgrand.enlive-html :as html]
   [bultitude.core :as b]
   [markdown.core :as md]
   [net.cgrand.enlive-html :as html]))

(def doco_charset "text/html; charset=utf-8")

(defn docs-by-ns
  "Return a list of docs by ns"
  [n meta-key]
  (when-let [ns (try
                  (the-ns n)
                  (catch java.lang.Exception e))]
    (let [publics (ns-publics ns)
          apis (reduce (fn [m [k v]]
                         (if-let [api-entry (-> v meta meta-key)]
                           (assoc m
                             k
                             (md/md-to-html-string api-entry))
                           m))
                       {}
                       publics)
          ns-meta (meta ns)
          title-kw (-> meta-key name (str "-section-title") keyword)
          section-title (if ns-meta (title-kw ns-meta) nil)]
      [section-title (merge apis (if ns-meta
                                   {"" (md/md-to-html-string (meta-key ns-meta))}
                                   {}))])))

(defn docs-map
  "Return a list of docs mapped by namespace"
  [ns-prefix meta-key]
  (let [ns-list (seq (b/namespaces-on-classpath :prefix ns-prefix))]
    (reduce
     (fn [m n]
       (if-let [[section-title docs] (docs-by-ns n meta-key)]
         (assoc m (or section-title (str n)) docs)
         m))
     {}
     ns-list)))

(defn uri-matches?
  [uri prefix]
  (re-find prefix uri))

(defmacro maybe-html-content
  ([expr] `(if-let [x# ~expr] (html/html-content x#) identity))
  ([expr & exprs] `(maybe-html-content (or ~expr ~@exprs))))

(html/defsnippet idx "doc/index.html" [:div#content]
  [{:keys [uri title date status version sections]}]
  [:.apiversion] (maybe-html-content version "1.0")
  [:h1#maintitle] (maybe-html-content title "API Documentation")
  [:span#pubdate] (maybe-html-content date (.toString (java.util.Date.)))
  [:span#pubstatus] (maybe-html-content status "DRAFT")
  [:li.doc-section] (html/clone-for
                     [s sections]
                     (html/do-> (html/html-content "<a href=\""
                                                   (str uri "/" s)
                                                   "\">"
                                                   s
                                                   "</a>"))))

(html/deftemplate layout "doc/layout.html"
  [{:keys [base-uri uri title date status version sections content]}]
  [:link#csslink] (html/set-attr :href (str base-uri "/doc.css"))
  [:title] (maybe-html-content title "")
  [:body] (maybe-html-content content ""))

(defn html-docs-for-section
  [section]
  (apply str (get (docs-map) section)))

(defn handle-doc-request
  [req section base-uri ns-prefix meta-key context-fn index-fn layout-fn]
  (let [doco (docs-map ns-prefix meta-key)
        context (merge (context-fn doco) {:base-uri base-uri
                                          :uri (:uri req)})]
    (if (= "" section)
      (let [context (assoc context :sections (keys doco))
            c (apply str (html/emit* (index-fn context)))
            context (merge context {:content c} )]
        (-> (response (apply str (layout-fn context)))
            (content-type doco_charset)))
      (let [d (get doco section)
            context (merge context {:content (apply str (vals d))} )]
        (-> (response (apply str (layout-fn context)))
            (content-type doco_charset))))))

(defn default-context
  [doc-map]
  {})

(def default-index idx)

(def default-layout layout)

(defn wrap-doctordoc
  "Intercept and handle documentation requests."
  [handler & {:keys [uri-prefix-regex
                     namespace-prefix
                     meta-key
                     context-fn
                     index-fn
                     layout-fn] :as options}]
  {:pre [(string? namespace-prefix)]}
  (fn [req]
    (if-let [matched (uri-matches? (:uri req) uri-prefix-regex)]
      (let [section (if (= (count (:uri req)) (count matched))
                      ""
                      (subs (:uri req) (+ 1 (count matched))))]
        (if (= "doc.css" section)
          (resource-response "doc/doc.css")
          (handle-doc-request req
                              section
                              (or matched (:uri req))
                              namespace-prefix
                              (or meta-key :api)
                              (or context-fn default-context)
                              (or index-fn default-index)
                              (or layout-fn default-layout))))
      (handler req))))
