(ns cljs-utils.core
  (:require [goog.events :as events]
            [clojure.string :as str]
            [cljs.core.async :as async :refer [<! >! put! take! chan]])
  (:import [goog Uri]
           [goog.Uri QueryData]))

(defn dev-mode?
  "Return boolean if development mode"
  ([]
   (let [url (.parse Uri js/window.location)
         domain (.getDomain url)]
     (or (str/starts-with? domain "localhost") (str/ends-with? domain ".ngrok-free.app"))))
  ([port]
    (let [url (.parse Uri js/window.location)]
      (or (dev-mode?) (= port (.getPort url)) ))))

(defn by-id
  "Short-hand for document.getElementById(id)"
  [id]
  (.getElementById js/document id))

(defn listen
  "listen on js events in a channel and return it"
  [el type]
  (let [out (chan)]
    (events/listen el type
      (fn [e] (put! out e)))
    out))

(defn set-interval
  "Invoke the given function after and every delay milliseconds."
  [delay f]
  (js/setInterval f delay))

(defn clear-interval
  "Cancel the periodic invokation specified by the given interval id."
  [interval-id]
  (js/clearInterval interval-id))


(defn events->chan
  "Given a target DOM element and event type return a channel of
  observed events. Can supply the channel to receive events as third
  optional argument."
  ([el event-type] (events->chan el event-type (chan)))
  ([el event-type c]
   (events/listen el event-type
     (fn [e] (put! c e)))
   c))

(defn mouse-loc->vec
  "Given a Google Closure normalized DOM mouse event return the
  mouse x and y position as a two element vector."
  [e]
  [(.-clientX e) (.-clientY e)])

(defn show!
  "Given a CSS id and a message string append a child paragraph element
  with the given message string."
  [id msg]
  (let [el (.getElementById js/document id)
        p  (.createElement js/document "p")]
    (set! (.-innerHTML p) msg)
    (.appendChild el p)))

(defn to-slug [str]
  (-> str
     (str/split #"\s")
     (#(str/join "-" %))
     (str/lower-case)))

(defn form-data [xs]
  (str (.createFromKeysValues QueryData (clj->js (keys xs)) (clj->js (vals xs)))))
