(ns facetious.authentication.shopify
  (:require [environ.core :refer [env]]
            [facetious.helpers :as helpers]
            [pandect.algo.sha256 :refer [sha256-hmac]]
            [liberator.core :refer [defresource log!]]
            [liberator.representation :refer [ring-response as-response]]
            [ring.util.response :as util]
            [clojure.data.json :as json]
            [clj-http.client :as client]
            [clojure.java.io :as io]))

;; Shopify does not use oauth-verifier, signatures are not verified

(defn make-consumer [domain] {:key (env :shopify-consumer-token)
                            :secret (env :shopify-consumer-secret)
                            :authorize-uri (str "https://" domain "/admin/oauth/authorize")
                            :access-uri (str "https://" domain "/admin/oauth/access_token")
                            :domain domain})

(defn app-store [{headers :headers session :session params :params :as req}]
  (let [valid-request? (fn [params]
                         (let [data (str "shop=" (:shop params) "&timestamp=" (:timestamp params))]
                           (= (:hmac params) (sha256-hmac data (env :shopify-consumer-secret)))))]
    (if (and (= (keys params) '(:hmac :shop :signature :timestamp)) (valid-request? params))
      (let [session (assoc session :shopify-domain (:shop params))]
        (-> (util/redirect (str "http://" (get headers "host")))
            (assoc :session session)))
      (util/redirect (str "http://" (get headers "host"))))))

(defn authorize [{headers :headers session :session params :params :as req}]
  (let [consumer (make-consumer (:domain params))
        redirect-uri (str (:authorize-uri consumer) "?client_id=" (env :shopify-consumer-token) "&scope=write_orders,read_customers,read_products,read_orders&redirect_uri=" (helpers/derive-callback headers "/shopify/callback"))]
    {:status 200
     :session (assoc session :shopify {:consumer consumer :redirect-uri redirect-uri}) 
     :body redirect-uri}))

(defn authenticate [{{query :params {{request-token :request-token consumer :consumer} :shopify} :session} :request}]
  (try 
    [true {::access-token-response (->
                                    (client/post (:access-uri consumer) {:form-params {:client_id (:key consumer)
                                                                                       :client_secret (:secret consumer)
                                                                                       :code (:code query)}})
                                    (:body)
                                    (json/read-str :key-fn keyword))}]
    (catch Exception e [false {::error (ex-data e)}])))

(defresource callback
  :available-media-types ["text/html"]
  :authorized? (fn [ctx]
             (authenticate ctx))
  :as-response (fn [d {{session :session} :request :as ctx}]
                 (let [domain (:domain (:consumer (:shopify session)))
                       session (assoc session :shopify (merge (ctx ::access-token-response) {:domain domain}))]
                   (-> (as-response d ctx)
                       (assoc :session session))))
  :handle-ok (fn [ctx] (-> (util/redirect "/")
                          (assoc :flash {:message "Successfully authorized by Shopify." :level :success})
                          (ring-response))))
