(ns facetious.authentication.google
  (:require [gapi.auth :as auth]
            [gapi.core :refer [call build]]
            [environ.core :refer [env]]
            [ring.util.response :as util]
            [clojure.tools.logging :as log]))

(def ^{:dynamic true :doc "Dynamic var used over the life of a Ring session."} *auth*)
(def scopes ["https://www.googleapis.com/auth/plus.me" "https://www.googleapis.com/auth/userinfo.email" "https://www.googleapis.com/auth/youtube"])
(def oauth-service #(build "https://www.googleapis.com/discovery/v1/apis/oauth2/v2/rest"))

(defn authorize [{session :session :as req}]
  (binding [*auth* (auth/create-auth (env :client-id) (env :client-secret) (env :redirect-url))]
    (let [url (auth/generate-auth-url *auth* scopes)
          auth {:auth *auth* :url url}]
      (log/debug "authorize " auth)
      (-> (util/response (:url auth))
          (assoc :session (assoc session :google auth))))))

(defn about-me
  "What I want from the Google+ profile"
  [{session :session :as req}]
  (let [auth (get-in session [:google :auth])
        userinfo (call auth (oauth-service) "oauth2.userinfo/get" {})]
    userinfo))

(defn authenticate
  "Authenticate user"
  [{session :session params :params :as req}]
  (let [auth (get-in session [:google :auth])]
    (auth/exchange-token auth (:code params) (:state params))
    (auth/is-valid auth)))

(defn callback [{session :session params :params :as req}]
  (log/debug "callback " session)
  (if (authenticate req)
    (let [me (about-me req)]
      (-> (util/redirect "/")
          (assoc :flash {:message (str (:given_name me) " successfully authorized by Google.") :level :success})
          (assoc :session (assoc session :me me))))
    (-> (util/redirect "/")
        (assoc :flash {:message "Could not authorize you (Google)" :level :warn}))))

;; not needed
(defn token-valid?
  "Checks if google+ token is valid (could implement an auto refresh)"
  [auth]
  (and auth (auth/is-valid auth)))

(defn token-info
  "Checks if google+ token is valid (could implement an auto refresh)"
  [auth]
  (println-str "Token expires in" (float (/ (- (:expires @auth) (System/currentTimeMillis)) 60000)) "minutes"))


