(ns winkel.gateway.paypal
  (:require [clj-http.client :as client]
            [clojure.core.memoize :as memo]
            [clojure.java.io :as io]
            [clojure.string :as str]
            [environ.core :refer [env]]
            [clojure.tools.logging :as log]
            [clojure.data.json :as json]))

(defn get-token
  ([]
   (get-token (env :paypal-endpoint) (:paypal-client env) (:paypal-secret env)))
  ([paypal-endpoint paypal-client paypal-secret]
   (-> (client/post (str paypal-endpoint "/v1/oauth2/token")
                    {:as :auto
                     :accept :json
                     :basic-auth [paypal-client paypal-secret]
                     :headers {"Accept-Language" "en_US"}
                     :form-params {:grant_type "client_credentials"}})
       :body)))

(def token (memo/ttl get-token :ttl/threshold (* 1000 30000)))

(defn approve [plan]
  (let [resp (client/post (str (:paypal-endpoint env) "/v1/payments/payment")
                     {:as :auto
                      :accept :json
                      :content-type :json
                      :headers {"Authorization" (str (:token_type (token)) " " (:access_token (token)))}
                      :form-params {:intent "sale"
                                    :redirect_urls {:return_url (:return-url plan) 
                                                    :cancel_url (str (:return-url plan) "-cancel")}
                                    :payer {:payment_method "paypal"}
                                    :transactions [{:amount {:total (:amount plan)
                                                             :currency (:currency plan)}
                                                    :description (:description plan)
                                                    :custom (pr-str {:plan_id (:id plan)})}]}})]
    (log/info "Approving payment for" (:uid plan) ", status" (:status resp))
    (:body resp)))

(defn execute [payment-id payer-id]
  (log/info "Executing payment for" payer-id)
  (let [resp (client/post (str (env :paypal-endpoint) "/v1/payments/payment/" payment-id "/execute/")
                            {:as :auto
                             :accept :json
                             :content-type :json
                             :headers {"Authorization" (str (:token_type (token)) " " (:access_token (token)))}
                             :form-params {:payer_id payer-id}
                             :throw-exceptions false})]
    (log/info "Payment executed with status" (:status resp))
    (if (= (:status resp) 200)
      (:body resp)
      (try
        (let [v (json/read-str (:body resp) :key-fn keyword)]
          (log/fatal "Paypal payment irregular return code" resp)
          {:state "error" :message (:message v) :name (:name v)})
        (catch Exception e {:state "error" :message (:status resp)})))))

(defn refund
  ([tx-id]
   (refund (env :paypal-endpoint) token tx-id))
  ([paypal-endpoint token-fn tx-id]
   (let [resp (client/post (str paypal-endpoint "/v1/payments/sale/" tx-id "/refund")
                           {:as :auto
                            :accept :json
                            :content-type :json
                            :headers {"Authorization" (str (:token_type (token-fn)) " " (:access_token (token-fn)))}
                            :form-params {}})]
     (:body resp))))

(defn show-refund
  ([refund-id]
   (show-refund (env :paypal-endpoint) token refund-id))
  ([ paypal-endpoint token-fn refund-id]
   (let [resp (client/get (str paypal-endpoint "/v1/payments/refund/" refund-id)
                          {:as :auto
                           :accept :json
                           :content-type :json
                           :headers {"Authorization" (str (:token_type (token-fn)) " " (:access_token (token-fn)))}
                           })]
     (:body resp))))

(defn show-payment
  ([payment-id]
   (show-payment (env :paypal-endpoint) token payment-id))
  ([paypal-endpoint token-fn payment-id]
   (let [resp (client/get (str paypal-endpoint "/v1/payments/payment/" payment-id)
                          {:as :auto
                           :accept :json
                           :content-type :json
                           :headers {"Authorization" (str (:token_type (token-fn)) " " (:access_token (token-fn)))}
                           })]
     (:body resp))))

(defn payments
  ([count]
   (payments (env :paypal-endpoint) token count))
  ([paypal-endpoint token-fn count]
   (let [resp (client/get (str paypal-endpoint "/v1/payments/payment")
                          {:as :auto
                           :accept :json
                           :content-type :json
                           :headers {"Authorization" (str (:token_type (token-fn)) " " (:access_token (token-fn)))}
                           :query-params {"count" count}})]
     (:body resp))))

(defn show-sale
  ([tx-id]
   (show-sale (env :paypal-endpoint) token tx-id))
  ([paypal-endpoint token-fn tx-id]
   (let [resp (client/get (str paypal-endpoint "/v1/payments/sale/" tx-id)
                          {:as :auto
                           :accept :json
                           :content-type :json
                           :headers {"Authorization" (str (:token_type (token-fn)) " " (:access_token (token-fn)))}
                           })]
     (:body resp))))

(defn show-payment
  ([payment-id]
   (show-payment (env :paypal-endpoint) token payment-id))
  ([paypal-endpoint token-fn payment-id]
   (let [resp (client/get (str paypal-endpoint "/v1/payments/payment/" payment-id)
                          {:as :auto
                           :accept :json
                           :content-type :json
                           :headers {"Authorization" (str (:token_type (token-fn)) " " (:access_token (token-fn)))}
                           })]
     (:body resp))))
