(ns winkel.gateway.paypal
  (:require [clj-http.client :as client]
            [clojure.core.memoize :as memo]
            [clojure.java.io :as io]
            [clojure.string :as str]
            [environ.core :refer [env]]
            [clojure.tools.logging :as log]
            [clojure.data.json :as json]))

(defn get-token
  ([]
   (get-token (env :paypal-endpoint) (:paypal-client env) (:paypal-secret env)))
  ([paypal-endpoint paypal-client paypal-secret]
   (-> (client/post (str paypal-endpoint "/v1/oauth2/token")
                    {:as :auto
                     :accept :json
                     :basic-auth [paypal-client paypal-secret]
                     :headers {"Accept-Language" "en_US"}
                     :form-params {:grant_type "client_credentials"}})
       :body)))

(def token (memo/ttl get-token :ttl/threshold (* 1000 30000)))

(defn approve [plan]
  (let [resp (client/post (str (:paypal-endpoint env) "/v1/payments/payment")
                     {:as :auto
                      :accept :json
                      :content-type :json
                      :headers {"Authorization" (str (:token_type (token)) " " (:access_token (token)))}
                      :form-params {:intent "sale"
                                    :redirect_urls {:return_url (:return-url plan) 
                                                    :cancel_url (str (:return-url plan) "-cancel")}
                                    :payer {:payment_method "paypal"}
                                    :transactions [{:amount {:total (:amount plan)
                                                             :currency (:currency plan)}
                                                    :description (:description plan)
                                                    :custom (pr-str {:plan_id (:id plan)})}]
                                    :experience_profile_id (:paypal-experience-id env)}})]
    (log/info "Approving payment" (:status resp))
    (:body resp)))

(defn execute [payment-id payer-id]
  (log/info "Executing payment for" payer-id)
  (let [resp (client/post (str (env :paypal-endpoint) "/v1/payments/payment/" payment-id "/execute/")
                            {:as :auto
                             :accept :json
                             :content-type :json
                             :headers {"Authorization" (str (:token_type (token)) " " (:access_token (token)))}
                             :form-params {:payer_id payer-id}
                             :throw-exceptions false})]
    (log/info "Payment executed with status" (:status resp))
    (if (= (:status resp) 200)
      (:body resp)
      (try
        (let [v (json/read-str (:body resp) :key-fn keyword)]
          (log/fatal "Paypal payment irregular return code" resp)
          {:state "error" :message (:message v) :name (:name v)})
        (catch Exception e {:state "error" :message (:status resp)})))))

(defn refund
  ([tx-id]
   (refund (env :paypal-endpoint) (token) tx-id))
  ([paypal-endpoint token tx-id]
   (let [resp (client/post (str paypal-endpoint "/v1/payments/sale/" tx-id "/refund")
                           {:as :auto
                            :accept :json
                            :content-type :json
                            :headers {"Authorization" (str (:token_type token) " " (:access_token token))}
                            :form-params {}})]
     (:body resp))))

(defn show-refund
  ([refund-id]
   (show-refund (env :paypal-endpoint) (token) refund-id))
  ([paypal-endpoint token refund-id]
   (let [resp (client/get (str paypal-endpoint "/v1/payments/refund/" refund-id)
                          {:as :auto
                           :accept :json
                           :content-type :json
                           :headers {"Authorization" (str (:token_type token) " " (:access_token token))}
                           })]
     (:body resp))))

(defn show-payment
  ([payment-id]
   (show-payment (env :paypal-endpoint) (token) payment-id))
  ([paypal-endpoint token payment-id]
   (let [resp (client/get (str paypal-endpoint "/v1/payments/payment/" payment-id)
                          {:as :auto
                           :accept :json
                           :content-type :json
                           :headers {"Authorization" (str (:token_type token) " " (:access_token token))}
                           })]
     (:body resp))))

(defn payments
  ([count]
   (payments (env :paypal-endpoint) (token) count))
  ([paypal-endpoint token count]
   (let [resp (client/get (str paypal-endpoint "/v1/payments/payment")
                          {:as :auto
                           :accept :json
                           :content-type :json
                           :headers {"Authorization" (str (:token_type token) " " (:access_token token))}
                           :query-params {"count" count}})]
     (:body resp))))

(defn show-sale
  ([tx-id]
   (show-sale (env :paypal-endpoint) (token) tx-id))
  ([paypal-endpoint token tx-id]
   (let [resp (client/get (str paypal-endpoint "/v1/payments/sale/" tx-id)
                          {:as :auto
                           :accept :json
                           :content-type :json
                           :headers {"Authorization" (str (:token_type token) " " (:access_token token))}
                           })]
     (:body resp))))

(defn show-payment
  ([payment-id]
   (show-payment (env :paypal-endpoint) (token) payment-id))
  ([paypal-endpoint token payment-id]
   (let [resp (client/get (str paypal-endpoint "/v1/payments/payment/" payment-id)
                          {:as :auto
                           :accept :json
                           :content-type :json
                           :headers {"Authorization" (str (:token_type token) " " (:access_token token))}
                           })]
     (:body resp))))

(defn list-profiles
  ([]
   (list-profiles (env :paypal-endpoint) (token)))
  ([paypal-endpoint token]
   (let [resp (client/get (str paypal-endpoint "/v1/payment-experience/web-profiles")
                          {:as :auto
                           :accept :json
                           :content-type :json
                           :headers {"Authorization" (str (:token_type token) " " (:access_token token))}
                           })]
     (:body resp))))

(defn create-profile
  ([profile]
   (create-profile (env :paypal-endpoint) (token) profile))
  ([paypal-endpoint token profile]
   (let [resp (client/post (str paypal-endpoint "/v1/payment-experience/web-profiles")
                            {:as :auto
                             :accept :json
                             :content-type :json
                             :headers {"Authorization" (str (:token_type token) " " (:access_token token))}
                             :form-params profile})]
     (:body resp))))

(defn show-profile
  ([profile-id]
   (show-profile (env :paypal-endpoint) (token) profile-id))
  ([paypal-endpoint token profile-id]
   (let [resp (client/get (str paypal-endpoint "/v1/payment-experience/web-profiles/" profile-id)
                          {:as :auto
                           :accept :json
                           :content-type :json
                           :headers {"Authorization" (str (:token_type token) " " (:access_token token))}
                           })]
     (:body resp))))

(defn update-profile
  ([profile-id profile]
   (update-profile (env :paypal-endpoint) (token) profile-id profile))
  ([paypal-endpoint token profile-id profile]
   (let [resp (client/put (str paypal-endpoint "/v1/payment-experience/web-profiles/" profile-id)
                            {:as :auto
                             :accept :json
                             :content-type :json
                             :headers {"Authorization" (str (:token_type token) " " (:access_token token))}
                             :form-params profile})]
     (:body resp))))

(defn list-plans
  ([status]
   {:pre [(some #{status} ["ACTIVE" "CREATED" "DELETED" "INACTIVE"])]}
   (list-plans (env :paypal-endpoint) (token) status))
  ([paypal-endpoint token status]
   {:pre [(some #{status} ["ACTIVE" "CREATED" "DELETED" "INACTIVE"])]}
   (let [resp (client/get (str paypal-endpoint "/v1/payments/billing-plans/")
                          {:as :auto
                           :accept :json
                           :content-type :json
                           :headers {"Authorization" (str (:token_type token) " " (:access_token token))}
                           :query-params {"status" status}})]
     (:body resp))))

(defn create-plan
  ([plan]
   (create-plan (env :paypal-endpoint) (token) plan))
  ([paypal-endpoint token plan]
   (let [resp (client/post (str paypal-endpoint "/v1/payments/billing-plans/")
                            {:as :auto
                             :accept :json
                             :content-type :json
                             :headers {"Authorization" (str (:token_type token) " " (:access_token token))}
                             :form-params plan})]
     (:body resp))))

(defn show-plan
  ([plan-id]
   (show-plan (env :paypal-endpoint) (token) plan-id))
  ([paypal-endpoint token plan-id]
   (let [resp (client/get (str paypal-endpoint "/v1/payments/billing-plans/" plan-id)
                          {:as :auto
                           :accept :json
                           :content-type :json
                           :headers {"Authorization" (str (:token_type token) " " (:access_token token))}
                           })]
     (:body resp))))

(defn update-plan
  ([plan-id plan-patch]
   (update-plan (env :paypal-endpoint) (token) plan-id plan-patch))
  ([paypal-endpoint token plan-id plan-patch]
   (let [resp (client/patch (str paypal-endpoint "/v1/payments/billing-plans/" plan-id)
                            {:as :auto
                             :accept :json
                             :content-type :json
                             :headers {"Authorization" (str (:token_type token) " " (:access_token token))}
                             :form-params plan-patch})]
     (:body resp))))

(defn create-agreement
  ([agreement]
   (create-agreement (env :paypal-endpoint) (token) agreement))
  ([paypal-endpoint token agreement]
   (let [resp (client/post (str paypal-endpoint "/v1/payments/billing-agreements/")
                            {:as :auto
                             :accept :json
                             :content-type :json
                             :headers {"Authorization" (str (:token_type token) " " (:access_token token))}
                             :form-params agreement})]
     (:body resp))))

(defn show-agreement
  ([agreement-id]
   (show-agreement (env :paypal-endpoint) (token) agreement-id))
  ([paypal-endpoint token agreement-id]
   (let [resp (client/get (str paypal-endpoint "/v1/payments/billing-agreements/" agreement-id)
                          {:as :auto
                           :accept :json
                           :content-type :json
                           :headers {"Authorization" (str (:token_type token) " " (:access_token token))}
                           })]
     (:body resp))))

(defn update-agreement
  ([agreement-id agreement-patch]
   (update-agreement (env :paypal-endpoint) (token) agreement-id agreement-patch))
  ([paypal-endpoint token agreement-id agreement-patch]
   (let [resp (client/patch (str paypal-endpoint "/v1/payments/billing-agreements/" agreement-id)
                            {:as :auto
                             :accept :json
                             :content-type :json
                             :headers {"Authorization" (str (:token_type token) " " (:access_token token))}
                             :form-params agreement-patch})]
     (:body resp))))

(defn bill-agreement-balance
  ([agreement-id amount]
   (bill-agreement-balance (env :paypal-endpoint) (token) agreement-id amount))
  ([paypal-endpoint token agreement-id amount]
   (let [resp (client/post (str paypal-endpoint "/v1/payments/billing-agreements/" agreement-id "/bill-balance")
                            {:as :auto
                             :accept :json
                             :content-type :json
                             :headers {"Authorization" (str (:token_type token) " " (:access_token token))}
                             :form-params amount})]
     (:body resp))))

(defn cancel-agreement
  ([agreement-id]
   (cancel-agreement (env :paypal-endpoint) (token) agreement-id))
  ([paypal-endpoint token agreement-id]
   (let [resp (client/post (str paypal-endpoint "/v1/payments/billing-agreements/" agreement-id "/cancel")
                            {:as :auto
                             :accept :json
                             :content-type :json
                             :headers {"Authorization" (str (:token_type token) " " (:access_token token))}})]
     (:body resp))))

(defn reactivate-agreement
  ([agreement-id]
   (reactivate-agreement (env :paypal-endpoint) (token) agreement-id))
  ([paypal-endpoint token agreement-id]
   (let [resp (client/post (str paypal-endpoint "/v1/payments/billing-agreements/" agreement-id "/reactivate")
                            {:as :auto
                             :accept :json
                             :content-type :json
                             :headers {"Authorization" (str (:token_type token) " " (:access_token token))}})]
     (:body resp))))

(defn set-agreement-balance
  ([agreement-id amount]
   (set-agreement-balance (env :paypal-endpoint) (token) agreement-id amount))
  ([paypal-endpoint token agreement-id amount]
   (let [resp (client/post (str paypal-endpoint "/v1/payments/billing-agreements/" agreement-id "/set-balance")
                            {:as :auto
                             :accept :json
                             :content-type :json
                             :headers {"Authorization" (str (:token_type token) " " (:access_token token))}
                             :form-params amount})]
     (:body resp))))

(defn suspend-agreement
  ([agreement-id]
   (suspend-agreement (env :paypal-endpoint) (token) agreement-id))
  ([paypal-endpoint token agreement-id]
   (let [resp (client/post (str paypal-endpoint "/v1/payments/billing-agreements/" agreement-id "/suspend")
                            {:as :auto
                             :accept :json
                             :content-type :json
                             :headers {"Authorization" (str (:token_type token) " " (:access_token token))}})]
     (:body resp))))

(defn list-agreement-transactions
  ([agreement-id start-date end-date]
   (list-agreement-transactions (env :paypal-endpoint) (token) agreement-id start-date end-date))
  ([paypal-endpoint token agreement-id start-date end-date]
   (let [resp (client/get (str paypal-endpoint "/v1/payments/billing-agreements/" agreement-id "/transactions")
                          {:as :auto
                           :accept :json
                           :content-type :json
                           :headers {"Authorization" (str (:token_type token) " " (:access_token token))}
                           :query-params {"start_date" start-date
                                          "end_date" end-date}})]
     (:body resp))))

(defn execute-agreement
  ([payment-token]
   (execute-agreement (env :paypal-endpoint) (token) payment-token))
  ([paypal-endpoint token payment-token]
   (let [resp (client/post (str paypal-endpoint "/v1/payments/billing-agreements/" payment-token "/agreement-execute")
                            {:as :auto
                             :accept :json
                             :content-type :json
                             :headers {"Authorization" (str (:token_type token) " " (:access_token token))}})]
     (:body resp))))

(defn create-webhook
  ([webhook]
   (create-webhook (env :paypal-endpoint) (token) webhook))
  ([paypal-endpoint token webhook]
   (let [resp (client/post (str paypal-endpoint "/v1/notifications/webhooks")
                            {:as :auto
                             :accept :json
                             :content-type :json
                             :headers {"Authorization" (str (:token_type token) " " (:access_token token))}
                             :form-params webhook})]
     (:body resp))))


(defn simulate-webhook-event
  ([webhook-id]
   (simulate-webhook-event (env :paypal-endpoint) (token) webhook-id))
  ([paypal-endpoint token webhook-id]
   (let [resp (client/post (str paypal-endpoint "/v1/notifications/simulate-event")
                            {:as :auto
                             :accept :json
                             :content-type :json
                             :headers {"Authorization" (str (:token_type token) " " (:access_token token))}
                             :form-params {:webhook_id webhook-id
                                           :event_type "PAYMENT.AUTHORIZATION.CREATED"}})]
     (:body resp))))

(defn show-webhook
  ([webhook-id]
   (show-webhook (env :paypal-endpoint) (token) webhook-id))
  ([paypal-endpoint token webhook-id]
   (let [resp (client/get (str paypal-endpoint "/v1/notifications/webhooks/" webhook-id )
                            {:as :auto
                             :accept :json
                             :content-type :json
                             :headers {"Authorization" (str (:token_type token) " " (:access_token token))}})]
     (:body resp))))

(defn list-webhooks
  ([]
   (list-webhooks (env :paypal-endpoint) (token)))
  ([paypal-endpoint token]
   (let [resp (client/get (str paypal-endpoint "/v1/notifications/webhooks/")
                            {:as :auto
                             :accept :json
                             :content-type :json
                             :headers {"Authorization" (str (:token_type token) " " (:access_token token))}})]
     (:body resp))))

(defn delete-webhook
  ([webhook-id]
   (delete-webhook (env :paypal-endpoint) (token) webhook-id))
  ([paypal-endpoint token webhook-id]
   (let [resp (client/delete (str paypal-endpoint "/v1/notifications/webhooks/" webhook-id )
                            {:as :auto
                             :accept :json
                             :content-type :json
                             :headers {"Authorization" (str (:token_type token) " " (:access_token token))}})]
     (:body resp))))

(defn update-webhook
  ([webhook-id webhook-patch]
   (update-webhook (env :paypal-endpoint) (token) webhook-id webhook-patch))
  ([paypal-endpoint token webhook-id webhook-patch]
   (let [resp (client/patch (str paypal-endpoint "/v1/notifications/webhooks/" webhook-id)
                            {:as :auto
                             :accept :json
                             :content-type :json
                             :headers {"Authorization" (str (:token_type token) " " (:access_token token))}
                             :form-params webhook-patch})]
     (:body resp))))

(defn show-event
  ([event-id]
   (show-event (env :paypal-endpoint) (token) event-id))
  ([paypal-endpoint token event-id]
   (let [resp (client/get (str paypal-endpoint "/v1/notifications/webhooks-events/" event-id )
                            {:as :auto
                             :accept :json
                             :content-type :json
                             :headers {"Authorization" (str (:token_type token) " " (:access_token token))}})]
     (:body resp))))

(defn list-events
  ([]
   (list-events (env :paypal-endpoint) (token)))
  ([paypal-endpoint token]
   (let [resp (client/get (str paypal-endpoint "/v1/notifications/webhooks-events")
                            {:as :auto
                             :accept :json
                             :content-type :json
                             :headers {"Authorization" (str (:token_type token) " " (:access_token token))}})]
     (:body resp))))

(defn resend-event
  ([event-id webhook-ids]
   (resend-event (env :paypal-endpoint) (token) event-id webhook-ids))
  ([paypal-endpoint token event-id webhook-ids]
   (let [resp (client/post (str paypal-endpoint "/v1/notifications/webhooks-events/" event-id "/resend")
                            {:as :auto
                             :accept :json
                             :content-type :json
                             :headers {"Authorization" (str (:token_type token) " " (:access_token token))}
                             :form-params webhook-ids})]
     (:body resp))))

(defn verify-webhook-signature
  ([data]
   (verify-webhook-signature (env :paypal-endpoint) (token) data))
  ([paypal-endpoint token data]
   (let [resp (client/post (str paypal-endpoint "/v1/notifications/verify-webhook-signature")
                            {:as :auto
                             :accept :json
                             :content-type :json
                             :headers {"Authorization" (str (:token_type token) " " (:access_token token))}
                             :form-params data})]
     (:body resp))))


