(ns winkel.gateway.stripe
  (:require  [clj-http.client :as client]
             [environ.core :refer [env]]
             [clojure.tools.logging :as log]
             [ring.util.response :refer [redirect]]))

(def api-endpoint "https://api.stripe.com/v1")

(defn session
  "All amounts sent to Stripe must be in integers, representing the lowest currency unit (e.g., cents). 9.99$ = 999.
  https://stackoverflow.com/questions/25546372/stripe-payment-gateway-i-wants-to-use-amount-in-decimal-format"
  [plan]
  (client/post (str api-endpoint "/checkout/sessions")
               {:as :auto
                :accept :json
                :basic-auth [(:stripe-secret env) ""]
                :multi-param-style :array
                :flatten-nested-form-params true
                :throw-exceptions false
                :form-params {:customer_email (:uid plan)
                              :client_reference_id (pr-str {:plan_id (:id plan) :uid (:uid plan)})
                              :success_url (:return-url plan)
                              :cancel_url (str (:return-url plan) "-cancel")
                              :payment_method_types ["card"]                                         
                              "line_items[]" {:name (:title plan)
                                              :description (:description plan)
                                              :amount (int (* 100 (:amount plan))) 
                                              :currency (:currency plan)
                                              :quantity 1}}}))


(defn source
  "Redirect is used for Alipay, not used in the case of WeChat."
  [plan]
  (client/post (str api-endpoint "/sources")
               {:as :auto
                :accept :json
                :basic-auth [(:stripe-secret env) ""]
                :throw-exceptions false
                :flatten-nested-form-params true
                :form-params {:type (name (:gateway plan))
                              :amount (int (* 100 (:amount plan)))
                              :currency (:currency plan)
                              :statement_descriptor (:title plan)
                              :owner {:email (:uid plan)}
                              :metadata {:plan (:id plan)}
                              :redirect {:return_url (:return-url plan)}}}))

(defn charge
  [source]
  (client/post (str api-endpoint "/charges")
                 {:as :auto
                  :accept :json
                  :basic-auth [(:stripe-secret env) ""]
                  :throw-exceptions true
                  :form-params {:amount (:amount source)
                                :currency (:currency source)
                                :source (:id source)} }))

(defn retrieve-event [event-id]
  (let [resp (client/get (str api-endpoint "/events/" event-id)
                         {:as :auto
                          :accept :json
                          :basic-auth [(:stripe-secret env) ""]})]
    (:body resp)))
