(ns dda.c4k-stats.core
  (:require
   [clojure.spec.alpha :as s]
   [orchestra.core :refer [defn-spec]]
   [dda.c4k-common.common :as cm]
   [dda.c4k-common.namespace :as ns]
   [dda.c4k-common.postgres :as pg]
   [dda.c4k-common.monitoring :as mon]
   [dda.c4k-common.yaml :as yaml]
   [dda.c4k-common.ingress :as ing]
   [dda.c4k-common.backup :as backup]
   [dda.c4k-stats.shynet :as shynet]
   [dda.c4k-stats.stats :as stats]))

(def config-defaults (merge
                      ing/default-config
                      {:namespace "stats"
                       :allowed-hosts ["*"]
                       :db-name "shynet"
                       :pv-storage-size-gb 5
                       :pvc-storage-class-name "local-path"
                       :postgres-image "postgres:16"
                       :postgres-size :2gb
                       :app-name "stats"
                       :backup-image "domaindrivenarchitecture/c4k-stats-backup:1.0.0-dev2025-04-15-16-19-59"
                       :backup-postgres true
                       :average-rate 200,
                       :burst-rate 400}))

(s/def ::config (s/keys :req-un [::ing/fqdn
                                 ::backup/restic-repository]
                        :opt-un [::ing/issuer
                                 ::ns/namespace
                                 ::shynet/allowed-hosts
                                 ::pg/db-name
                                 ::pg/pv-storage-size-gb
                                 ::pg/pvc-storage-class-name
                                 ::mon/mon-cfg]))

(s/def ::auth (s/keys :req-un [::shynet/django-secret-key
                               ::pg/postgres-db-user
                               ::pg/postgres-db-password
                               ::backup/restic-password
                               ::backup/aws-access-key-id
                               ::backup/aws-secret-access-key]
                      :opt-un [::backup/restic-new-password
                               ::mon/mon-auth]))

(s/def ::config-select (s/* #{"auth" "deployment"}))

(defn-spec config-objects seq?
  [config-select ::config-select
   config ::config]
  (let [resolved-config (merge config-defaults config)
        {:keys [fqdn]} resolved-config
        config-parts (if (empty? config-select)
                       ["auth" "deployment"]
                       config-select)]
    (map yaml/to-string
         (if (some #(= "deployment" %) config-parts)
           (cm/concat-vec
            (ns/generate resolved-config)
            (stats/config-objects resolved-config)
            (pg/config-objects resolved-config)
            (shynet/config-objects resolved-config)
            (ing/config-objects (merge
                                 {:service-name "stats"
                                  :service-port 80
                                  :fqdns [(str "plaus." fqdn)]}
                                 resolved-config))
            (ing/config-objects (merge
                                 {:service-name "shynet-webserver-service"
                                  :service-port 8080
                                  :fqdns [(str "shy." fqdn)]}
                                 resolved-config))
            (when (contains? resolved-config :restic-repository)
              (backup/config-objects resolved-config))
            (when (contains? resolved-config :mon-cfg)
              (mon/config-objects (:mon-cfg resolved-config))))
           []))))

(defn-spec auth-objects seq?
  [config-select ::config-select
   config ::config
   auth ::auth]
  (let [resolved-config (merge config-defaults config)
        config-parts (if (empty? config-select)
                       ["auth" "deployment"]
                       config-select)]
    (map yaml/to-string
         (if (some #(= "auth" %) config-parts)
           (cm/concat-vec
            (cm/concat-vec
             (pg/auth-objects resolved-config auth)
             (shynet/auth-objects resolved-config auth)
             (when (contains? resolved-config :restic-repository)
               (backup/auth-objects resolved-config auth))
             (cm/concat-vec
              (when (and (contains? auth :mon-auth) (contains? resolved-config :mon-cfg))
                (mon/auth-objects (:mon-cfg resolved-config) (:mon-auth auth))))))
           []))))
