; Licensed to the Apache Software Foundation (ASF) under one
; or more contributor license agreements. See the NOTICE file
; distributed with this work for additional information
; regarding copyright ownership. The ASF licenses this file
; to you under the Apache License, Version 2.0 (the
; "License"); you may not use this file except in compliance
; with the License. You may obtain a copy of the License at
;
; http://www.apache.org/licenses/LICENSE-2.0
;
; Unless required by applicable law or agreed to in writing, software
; distributed under the License is distributed on an "AS IS" BASIS,
; WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
; See the License for the specific language governing permissions and
; limitations under the License.

(ns org.domaindrivenarchitecture.pallet.crate.backup
   (:require
    [pallet.actions :as actions]
    [pallet.stevedore :as stevedore]
    [org.domaindrivenarchitecture.cm.util :as util]
    ))

(def facility
  :dda-backup)

(def backup-user-name 
  "dataBackupSource")

(defn- script-path
  [app-name]
  (str "/usr/lib/" app-name)
  )

(defn- cron-name
  [semantic-name script-type]
  (case script-type
      :backup 
      (str semantic-name "_backup")
      :restore 
      (str semantic-name "_restore")
      :source-transport
      (str semantic-name "_source_transport")
      )
  )

(defn- script-name
  [semantic-name script-type]
  (str (cron-name semantic-name script-type) ".sh")
  )


(defn- script-path-with-name
  [app-name semantic-name script-type]
  (str (script-path app-name) "/" (script-name semantic-name script-type))
  )

(defn create-backup-source-user
  []
  (actions/user backup-user-name 
                :action :create 
                :create-home true 
                :shell :bash
                :password "WIwn6jIUt2Rbc")
  (actions/directory (str "/home/" backup-user-name "/transport-outgoing")
                     :action :create
                     :owner backup-user-name
                     :group backup-user-name)
  (actions/directory (str "/home/" backup-user-name "/store")
                     :action :create
                     :owner backup-user-name
                     :group backup-user-name)
  (actions/directory (str "/home/" backup-user-name "/restore")
                     :action :create
                     :owner backup-user-name
                     :group backup-user-name)
  )

(defn create-source-environment
  [app-name]
  (actions/directory 
    (script-path app-name)
    :action :create
    :owner "root"
    :group "root")
  )

(defn create-source-backup
  [app-name semantic-name backup-lines]
  (actions/remote-file
    (script-path-with-name app-name semantic-name :backup)
    :mode "700"
    :overwrite-changes true
    :literal true
    :content (util/create-file-content 
               backup-lines))
  (actions/symbolic-link 
    (script-path-with-name app-name semantic-name :backup)
    (str "/etc/cron.daily/10_" (cron-name semantic-name :backup))
    :action :create)
  )

(defn create-source-restore
  [app-name semantic-name restore-lines]
  (actions/remote-file
    (script-path-with-name app-name semantic-name :restore)  
    :mode "700"
    :overwrite-changes true
    :literal true
    :content (util/create-file-content 
               restore-lines))
  )

(defn create-source-transport
  [app-name semantic-name source-transport-lines]
  (actions/remote-file
    (script-path-with-name app-name semantic-name :source-transport)   
    :mode "700"
    :overwrite-changes true
    :literal true
    :content (util/create-file-content 
               source-transport-lines))
  (actions/symbolic-link 
    (script-path-with-name app-name semantic-name :source-transport) 
    (str "/etc/cron.daily/20_" (cron-name semantic-name :source-transport))
    :action :create)
  )


(defn install-backup-environment
  [& {:keys [app-name]}]
  (let [installed-marker 
        (str "/home/pallet/state/" (name facility))]
    (actions/plan-when-not 
      (stevedore/script (file-exists? installed-marker))
      (create-backup-source-user)
      (create-source-environment app-name)
      (actions/file
        installed-marker
        :action :create)
      )))

(defn install-backup-app-instance
  [& {:keys [app-name
             instance-name
             backup-lines 
             source-transport-lines 
             restore-lines]}]
  (let [installed-marker 
        (str "/home/pallet/state/" (name facility) "-" instance-name)]
    (actions/plan-when-not 
      (stevedore/script (file-exists? installed-marker))
      (create-source-backup app-name instance-name backup-lines)
      (create-source-transport app-name instance-name source-transport-lines)
      (create-source-restore app-name instance-name restore-lines)
      (actions/file
        (str "/home/pallet/state/" (name facility) "-" instance-name)
        :action :create)
      )))