(ns dda.build.terragrunt
  (:require [orchestra.core :refer [defn-spec]]
            [clojure.spec.test.alpha :as st]
            [cheshire.core :refer [parse-string]]
            [dda.build.devops :as d]
            [dda.build.terragrunt.domain :as domain]
            [dda.build.infrastructure :as i]))

(st/instrument `clean-build-dir!)

(def default
  (merge d/default {:autoapply false
                    :tg-output-filenname "tg-out.json"}))

(defn-spec clean-build-dir! nil?
  [devops ::d/devops]
  (let [config (merge default devops)]
    (i/execute! (domain/clean-build-dir-command config) config)))

(defn-spec copy-terragrunt! nil?
  [devops ::d/devops]
  (let [config (merge default devops)]
    (doseq [c (domain/copy-terragrunt-command config)]
      (i/execute! c config))))

(defn-spec terragrunt-plan! nil?
  [devops ::d/devops]
  (let [config (merge default devops)]
    (doseq [c (domain/terragrunt-plan-command config)]
      (i/execute! c config))))

(defn-spec terragrunt-apply! nil?
  [devops ::d/devops]
  (let [config (merge default devops)]
    (doseq [c (domain/terragrunt-apply-command config)]
      (i/execute! c config))))

(defn-spec terragrunt-output! map?
  [devops ::d/devops]
  (let [config (merge default devops)]
    (doseq [c (domain/terragrunt-output-command config)]
      (i/execute! c config))
    (parse-string (slurp (domain/output-path config))
                  (fn [k] (keyword (.toLowerCase k))))))

(defn-spec terragrunt-destroy! nil?
  [devops ::d/devops]
  (let [config (merge default devops)]
    (doseq [c (domain/terragrunt-destroy-command config)]
      (i/execute! c config))))

(defn-spec plan! nil?
  [devops ::d/devops]
  (let [config (merge default devops)]
    (clean-build-dir! config)
    (d/create-build-dir! config)
    (copy-terragrunt! config)
    (terragrunt-plan! config)))

(defn-spec apply! map?
  [devops ::d/devops]
  (let [config (merge default devops)]
    (clean-build-dir! config)
    (d/create-build-dir! config)
    (copy-terragrunt! config)
    (terragrunt-apply! config)
    (terragrunt-output! config)))

(defn-spec destroy! nil?
  [devops ::d/devops]
  (let [config (merge default devops)]
    (clean-build-dir! config)
    (d/create-build-dir! config)
    (copy-terragrunt! config)
    (terragrunt-destroy! config)))

(st/instrument `clean-build-dir!)
(st/instrument `copy-terragrunt!)
(st/instrument `terragrunt-plan!)
(st/instrument `terragrunt-apply!)
(st/instrument `terragrunt-destroy!)
