(ns dda.build.c4k
  (:require [orchestra.core :refer [defn-spec]]
            [clojure.spec.test.alpha :as st]
            [cheshire.core :refer [parse-string generate-string]]
            [dda.build.devops :as d]
            [dda.build.c4k.domain :as domain]
            [dda.build.infrastructure :as i]))

(st/instrument `clean-build-dir!)

(def default
  (merge d/default {:autoapply false
                    :c4k-output-filename "c4k-app.yaml"
                    :c4k-config-filename "c4k-config.yaml"
                    :c4k-auth-filename "c4k-auth.yaml"}))

(defn-spec clean-build-dir! nil?
  [devops ::d/devops]
  (let [config (merge default devops)]
    (i/execute! (domain/clean-build-dir-command config) config)))

(defn-spec run-c4k-jar! nil?
  [devops ::d/devops]
  (let [config (merge default devops)]
    (doseq [c (domain/c4k-uberjar-command config)]
      (i/execute! c config))))

(defn-spec run-c4k-executable! nil?
  [devops ::d/devops]
  (let [config (merge default devops)]
    (doseq [c (domain/c4k-graalvm-command config)]
      (i/execute! c config))))

(defn-spec generate-jar! nil?
  [devops ::d/devops]
  (let [config (merge default devops)]
    (clean-build-dir! config)
    (d/create-build-dir! config)
    ; Maybe copy some default config+auth
    ; (copy-terragrunt! config)
    (run-c4k-jar! config)))

(defn-spec generate! nil?
  [devops ::d/devops]
  (let [config (merge default devops)]
    (clean-build-dir! config)
    (d/create-build-dir! config)
    ; Maybe copy some default config+auth
    ; (copy-terragrunt! config)
    (run-c4k-executable! config)))


(defn-spec insert-tf-out! nil?
  [devops ::d/devops
   tf-out ::domain/tf-out]
  (let [config (merge default devops)
        {:keys [stage]} config
        issuer (if (= stage "prod") "prod" "staging")
        fqdn (:fqdn (:value (:out tf-out)))
        default-c4k-config (parse-string (slurp (domain/config-path config))
                                     (fn [k] (keyword (.toLowerCase k))))]
    (->> default-c4k-config
         (merge {:issuer issuer :fqdn fqdn})
         (generate-string)
         (spit (domain/output-path config)))))


(st/instrument `clean-build-dir!)
(st/instrument `run-c4k-jar!)
(st/instrument `run-c4k-executable!)
(st/instrument `generate-jar!)
(st/instrument `generate!)
(st/instrument `insert-tf-out!)
