(ns dda.build.c4k
  (:require [orchestra.core :refer [defn-spec]]
            [clojure.spec.test.alpha :as st]
            [cheshire.core :refer [parse-string generate-string]]
            [dda.build.devops :as d]
            [dda.build.c4k.domain :as domain]
            [dda.build.infrastructure :as i]))

(st/instrument `clean-build-dir!)

(def default
  (merge d/default {:autoapply false
                    :c4k-output-filename "c4k-app.yaml"
                    :c4k-config-filename "c4k-config.yaml"
                    :c4k-auth-filename "c4k-auth.yaml"}))

(defn-spec clean-build-dir! nil?
  [devops ::d/devops]
  (let [config (merge default devops)]
    (i/execute! (domain/clean-build-dir-command config) config)))

(defn-spec run-c4k-jar! nil?
  [devops ::d/devops]
  (let [config (merge default devops)]
    (doseq [c (domain/c4k-uberjar-command config)]
      (i/execute! c config))))

(defn-spec run-c4k-executable! nil?
  [devops ::d/devops]
  (let [config (merge default devops)]
    (doseq [c (domain/c4k-graalvm-command config)]
      (i/execute! c config))))

; TODO: Generate functions assume that files have already been copied,
;       which will happen if this is run after terragrunt
;       but it is not guaranteed
(defn-spec generate-jar! nil?
  "Generates c4k app yaml using 'c4k-{module}-standalone.jar'
   Stores the result in 'c4k-app.yaml'
   
   Defaults: c4k-config.yaml c4k-auth.yaml c4k-app.yaml
   can be changed by adding another value for ':c4k-config-filename', ':c4k-auth-filename', ':c4k-output-filename'
  "
  [devops ::d/devops]
  (let [config (merge default devops)]
    (run-c4k-jar! config)))

(defn-spec generate! nil?
  "Generates c4k app yaml using 'c4k-{module}' (graalvm executable)
   Stores the result in 'c4k-app.yaml'
   
   Defaults: c4k-config.yaml c4k-auth.yaml c4k-app.yaml
   can be changed by adding another value for ':c4k-config-filename', ':c4k-auth-filename', ':c4k-output-filename'
  "
  [devops ::d/devops]
  (let [config (merge default devops)]
    (run-c4k-executable! config)))

(defn-spec insert-tf-out! nil?
  "Inserts relevant values from the tf output into the c4k config
   
   Default: c4k-config.yaml
   can be changed by adding another value for ':c4k-config-filename'
   "
  [devops ::d/devops
   tf-out ::domain/tf-out]
  (let [config (merge default devops)
        default-c4k-config (parse-string (slurp (domain/config-path config))
                                         (fn [k] (keyword (.toLowerCase k))))
        tf-out-c4k-config (domain/create-c4k-config config tf-out)]
    (->> default-c4k-config
         (merge tf-out-c4k-config)
         (generate-string)
         (spit (domain/config-path config)))))


(st/instrument `clean-build-dir!)
(st/instrument `run-c4k-jar!)
(st/instrument `run-c4k-executable!)
(st/instrument `generate-jar!)
(st/instrument `generate!)
(st/instrument `insert-tf-out!)
